/*
 * Rayngin - 3D 6DF framework/engine for approach&click quests in rectangular chambers with objects consisting of balls
 * Copyright (c) 2021 Sunkware
 * PubKey FP: 6B6D C8E9 3438 6E9C 3D97  56E5 2CE9 A476 99EF 28F6
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * [ WWW: sunkware.org ]                         [ E-MAIL: sunkware@gmail.com ]
 */

use std::f64::consts::PI;

use crate::base::{
    Attitude,
    Orientation,
    Position
};

#[derive(Clone, PartialEq, Debug)]
pub struct TimedAttitude {
    time: i128,
    att: Attitude
}

#[derive(Clone, PartialEq, Debug)]
pub struct Trajectory {
    id: String,
    pos: Position,
    timed_attitudes: Vec<TimedAttitude>,
    looped: bool
}

impl TimedAttitude {
    pub fn new(time: i128, x: i64, y: i64, z: i64, yaw: f64, pitch: f64, roll: f64) -> TimedAttitude {
        TimedAttitude {
            time,
            att: Attitude::new(
                x << 0x10, y << 0x10, z << 0x10,
                yaw * PI / 180.0, pitch * PI / 180.0, roll * PI / 180.0
            )
        }
    }

}

impl Trajectory {
    pub fn new(id: impl ToString, x: i64, y: i64, z: i64, looped: bool) -> Trajectory {
        let id = id.to_string();
        let timed_attitudes = Vec::<TimedAttitude>::new();

        Trajectory{id, pos: Position::new(x << 0x10, y << 0x10, z << 0x10), timed_attitudes, looped}
    }

    pub fn dup(&self, id: impl ToString, x: i64, y: i64, z: i64) -> Trajectory {
        Trajectory{id: id.to_string(), pos: Position::new(x << 0x10, y << 0x10, z << 0x10), timed_attitudes: self.timed_attitudes.clone(), looped: self.looped}
    }

    pub fn scale(&mut self, mult_spatial: f64, mult_temporal: f64) {
        let ms = (mult_spatial * 65536.0) as i64;
        let mt = (mult_temporal * 65536.0) as i64;
        for tatt in &mut self.timed_attitudes {
            tatt.att.pos = tatt.att.pos.scaled_noverflow(ms);
            tatt.time = (tatt.time * (mt as i128)) >> 0x10;
        }
    }

    pub fn id(&self) -> &String {
        &(self.id)
    }

    pub fn is_empty(&self) -> bool {
        self.timed_attitudes.is_empty()
    }

    pub fn first_time(&self) -> i128 {
        match self.timed_attitudes.first() {
            Some(first) => first.time,
            None => 0
        }
    }

    pub fn last_time(&self) -> i128 {
        match self.timed_attitudes.last() {
            Some(last) => last.time,
            None => 0
        }
    }

    pub fn first_attitude(&self) -> Attitude {
        match self.timed_attitudes.first() {
            Some(first) => Attitude{pos: self.pos.added(&(first.att.pos)), ori: first.att.ori.clone()},
            None => Attitude{pos: self.pos.clone(), ori: Orientation::new(0.0, 0.0, 0.0)}
        }
    }

    pub fn last_attitude(&self) -> Attitude {
        match self.timed_attitudes.last() {
            Some(last) => Attitude{pos: self.pos.added(&(last.att.pos)), ori: last.att.ori.clone()},
            None => Attitude{pos: self.pos.clone(), ori: Orientation::new(0.0, 0.0, 0.0)}
        }
    }

    pub fn add_timed_attitude(&mut self, tatt: TimedAttitude) -> Result<(), String> {
        match self.timed_attitudes.last() {
            Some(last) => {
                if tatt.time > last.time {
                    self.timed_attitudes.push(tatt);
                    Ok(())
                } else {
                    Err(format!("cannot add timed attitude: time {} is not greater than the latest time {}", tatt.time, last.time))
                }
            },
            None => {
                self.timed_attitudes.push(tatt);
                Ok(())
            }
        }
    }

    pub fn attitude(&self, time: i128) -> Attitude {
        let timed_attitudes = & self.timed_attitudes;

        let mut att = match timed_attitudes.last() {
            Some(last) => {
                let time = match self.looped {
                    false => time,
                    true => time % last.time
                };

                let first = timed_attitudes.first().unwrap();

                if time < first.time {
                    first.att.clone()
                } else if time >= last.time {
                    last.att.clone()
                } else { // "moving"...
                    // binary search
                    let (mut start, mut end) = (0, timed_attitudes.len() - 1);
                    while end > start + 1 {
                        let mid = (start + end) >> 1;
                        if time >= timed_attitudes[mid].time {
                            start = mid;
                        } else {
                            end = mid;
                        }
                    }
                    // now end = start + 1 and tatt[start].time <= time < tatt[end].time, segment of trajectory is found
                    let (start, end) = (&(timed_attitudes[start]), &(timed_attitudes[end]));
                    let delta = end.time - start.time;
                    let u: f64 = match delta {
                        0 => 0.0,
                        _ => ((time - start.time) as f64) / (delta as f64)
                    };
                    let w1 = u * u * (3.0 - 2.0 * u);
                    let w0 = 1.0 - w1;  // cubic interpolation; weight(0) = 1, w(1) = 0, w'(0) = w'(1) = 0
                    let w0_fpa = (w0 * 65536.0) as i128;
                    let w1_fpa = (w1 * 65536.0) as i128;
                    Attitude::new(
                        (((start.att.pos.x as i128) * w0_fpa + (end.att.pos.x as i128) * w1_fpa) >> 0x10) as i64,
                        (((start.att.pos.y as i128) * w0_fpa + (end.att.pos.y as i128) * w1_fpa) >> 0x10) as i64,
                        (((start.att.pos.z as i128) * w0_fpa + (end.att.pos.z as i128) * w1_fpa) >> 0x10) as i64,
                        start.att.ori.yaw * w0 + end.att.ori.yaw * w1,
                        start.att.ori.pitch * w0 + end.att.ori.pitch * w1,
                        start.att.ori.roll * w0 + end.att.ori.roll * w1
                    )
                }
            },
            None => Attitude::new(0, 0, 0, 0.0, 0.0, 0.0)
        };

        att.pos = att.pos.added(&(self.pos));

        att
    }

}
