/*
 * Rayngin - 3D 6DF framework/engine for approach&click quests in rectangular chambers with objects consisting of balls
 * Copyright (c) 2021 Sunkware
 * PubKey FP: 6B6D C8E9 3438 6E9C 3D97  56E5 2CE9 A476 99EF 28F6
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * [ WWW: sunkware.org ]                         [ E-MAIL: sunkware@gmail.com ]
 */

use crate::base;

use crate::sys::Image;

const QUARTSCALE_DIST_SHIFT: i64 = 0x2000; // 0x4000; // more to decrease distance averaging (near)
const QUARTSCALE_DIST_BINLOG_SHIFT: i32 = 8; // more to decrease distance averaging (far)

pub struct Texture {
    id: String,
    size_binlog: i32, // log_2 of size of images[0]
    size_binmasks: Vec<i32>,
    images: Vec<Image> // images[i+1] is quarterized image[i], 2^m x 2^m original to 1 x 1.
}

/// Consists of "cells" - usual Textures;
/// index of Texture is taken from array, pseudorandomly - see cell() - based on coordinates.
/// To increase "variety" of chamber faces...
pub struct SupraTexture {
    id: String,
    cell_size_binlog: u32,
    seed: u64,
    txr: [usize; 0x10]
}

impl Texture {
    pub fn load(base_filepath: impl ToString, id: impl ToString) -> Result<Texture, String> {
        let base_filepath = base_filepath.to_string();
        let id = id.to_string();

        match Image::load(base_filepath) {
            Ok(img) => {
                if img.width() != img.height() {
                    return Err(format!("image {}x{} is not square", img.width(), img.height()))
                }
                if !base::is_2pow(img.width() as i64) {
                    return Err(format!("image size {} is not a power of 2", img.width()))
                }

                let size_binlog: i32 = base::binlog(img.width() as i64);

                let mut images: Vec<Image> = Vec::new();
                let mut size_binmasks: Vec<i32> = Vec::new();

                images.push(img);
                size_binmasks.push((1i32 << size_binlog) - 1);
                for i in 1..=size_binlog { // including size_binlog. E.g. if it's 2, store original texture (size = 4), halved (size = 2), and quartered (size = 1)
                    images.push(images[(i as usize) - 1].quarterized());
                    size_binmasks.push((1i32 << (size_binlog - i)) - 1);
                }

                Ok(Texture{id, size_binlog, size_binmasks, images})
            },
            Err(s) => Err(s)
        }
    }

    pub fn id(&self) -> &String {
        &(self.id)
    }

    #[inline(always)]
    pub fn pixel(&self, x: i64, y: i64, dist: i64, rotm: u8) -> [u8; 4] { // texture is "infinite"
        let (mut x, mut y) = match rotm {
            0 => (x, y),
            1 => (y, -x),
            2 => (-x, -y),
            _ => (-y, x)
        };
        let dist = dist >> 0x10; // "integer" part
        let scale_binlog = (base::binlog(dist - QUARTSCALE_DIST_SHIFT) - QUARTSCALE_DIST_BINLOG_SHIFT).max(0).min(self.size_binlog) as usize;
        let size_binmask = self.size_binmasks[scale_binlog];
        x >>= scale_binlog; y >>= scale_binlog;
        self.images[scale_binlog].pixel_nocheck(((x >> 0x10) as i32) & size_binmask, ((y >> 0x10) as i32) & size_binmask)
        // let size_binmask = self.size_binmasks[0];
        // self.images[0].pixel_nocheck((x as i32) & size_binmask, (y as i32) & size_binmask)
    }

    #[inline(always)]
    pub fn pixel_smooth(&self, x: i64, y: i64, dist: i64, rotm: u8) -> [u8; 4] { // texture is "infinite"
        let (mut x, mut y) = match rotm {
            0 => (x, y),
            1 => (y, -x),
            2 => (-x, -y),
            _ => (-y, x)
        };
        let dist = dist >> 0x10; // "integer" part
        let scale_binlog = (base::binlog(dist - QUARTSCALE_DIST_SHIFT) - QUARTSCALE_DIST_BINLOG_SHIFT).max(0).min(self.size_binlog) as usize;
        let size_binmask = self.size_binmasks[scale_binlog];
        let image = &(self.images[scale_binlog]);

        x >>= scale_binlog; y >>= scale_binlog;

        let (x0, y0) = (((x >> 0x10) as i32) & size_binmask, ((y >> 0x10) as i32) & size_binmask);
        let (x1, y1) = ((x0 + 1) & size_binmask, (y0 + 1) & size_binmask);
        let (u, v) = (((x >> 8) as i32) & 0xFF, ((y >> 8) as i32) & 0xFF);
        let (nu, nv) = (0xFF - u, 0xFF - v);

        let c00 = image.pixel_nocheck(x0, y0);
        let c10 = image.pixel_nocheck(x1, y0);
        let c01 = image.pixel_nocheck(x0, y1);
        let c11 = image.pixel_nocheck(x1, y1);

        let mut color: [u8; 4] = [0; 4];
        for i in 0..3 {
            color[i] = (( ((c00[i] as i32) * nu + (c10[i] as i32) * u) * nv + ((c01[i] as i32) * nu + (c11[i] as i32) * u) * v ) >> 0x10) as u8;
        }

        color
    }

}

impl SupraTexture {
    pub fn new(id: impl ToString, cell_size_binlog: u32, seed: u64, txr: &Vec<usize>) -> SupraTexture {
        let id = id.to_string();

        let cell_size_binlog = cell_size_binlog + 0x10;

        let mut txr_rep: [usize; 0x10] = [0; 0x10];
        for i in 0..0x10 {
            txr_rep[i] = txr[i % txr.len()]
        }

        SupraTexture{id, cell_size_binlog, seed, txr: txr_rep}
    }

    pub fn id(&self) -> &String {
        &(self.id)
    }

    #[inline(always)]
    pub fn cell(&self, x: i64, y: i64) -> (usize, u8) {
        let (x, y) = ((x >> self.cell_size_binlog) as u64, (y >> self.cell_size_binlog) as u64);

        // "Pseudorandomness"... visible patterns should not occur

        let (seed, x, y) = (self.seed, x + 14159265, y + 35897932); // PI = 3.14159265 35897932+
        // 2 non-linearities of different kind
        let (u, v) = (x ^ y, (x ^ seed) * (y ^ seed));
        // mixing
        let (u, v) = (u ^ (v >> 12), v ^ (u >> 16));
        let (u, v) = (u ^ (v >> 6), v ^ (u >> 8));
        let (u, v) = (u ^ (v >> 3), v ^ (u >> 4));
        let prnd = ((u ^ v) & 0x3F) as usize;

        (self.txr[prnd >> 2], (prnd & 3) as u8) // 4 high bits - texture index, 2 low bits - rotation
    }
}
