/*
 * Rayngin - 3D 6DF framework/engine for approach&click quests in rectangular chambers with objects consisting of balls
 * Copyright (c) 2021 Sunkware
 * PubKey FP: 6B6D C8E9 3438 6E9C 3D97  56E5 2CE9 A476 99EF 28F6
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * [ WWW: sunkware.org ]                         [ E-MAIL: sunkware@gmail.com ]
 */

extern crate serde;
extern crate serde_json;

use std::{
    collections::HashMap,
    fs
};

use serde_json::Value;

use crate::base::{
    BASE_DIR,
    JEXT,
    PrependErrorString
};

use crate::sys::System;

const LINGUA_DIR: &str = "lingua";

pub struct Lingua {
    lang: String,
    lingua_map: HashMap<String, String>, // id to text in lang language
}

impl Lingua {
    pub fn new(sys: &System) -> Self {
        let lang = sys.lang();
        let lingua_map = HashMap::<String, String>::new();
        Self{lang, lingua_map}
    }

    pub fn load(&mut self, lingua_filepaths_noext: &[impl ToString]) -> Result<(), String> {
        for lingua_filepath_noext in lingua_filepaths_noext {
            let filepath = format!("{}/{}/{}/{}.{}", BASE_DIR, LINGUA_DIR, & self.lang, lingua_filepath_noext.to_string(), JEXT);
            let b = fs::read(&filepath).pre_err(format!("cannot read '{}'", &filepath))?;
            let v: Value = serde_json::from_slice(&b).pre_err(format!("cannot parse '{}'", &filepath))?;

            match v.as_object() {
                Some(objmap) => {
                    for (id, val) in objmap {
                        let id = id.clone();
                        match val.as_str() {
                            Some(text) => {
                                let text = text.to_string();
                                match self.lingua_map.insert(id.clone(), text.clone()) {
                                    Some(old_text) => return Err(format!("cannot load '{0}': cannot add (id '{1}' -> text '{2}'), because (id '{1}' -> text '{3}') already exists", &filepath, &id, &text, &old_text)),
                                    None => ()
                                };
                            },
                            None => return Err(format!("cannot load '{}': value for '{}' key is not a string", &filepath, &id))
                        }
                    }
                },
                None => return Err(format!("cannot load '{}': content is not a json object", &filepath))
            }
        }


        Ok(())
    }

    pub fn get(&self, id: impl ToString) -> Option<String> {
        let id = id.to_string();
        self.lingua_map.get(&id).map(|text| text.clone())
    }

    pub fn get_or_echo(&self, id: impl ToString) -> String {
        let id = id.to_string();
        self.get(&id).unwrap_or_else(|| format!("{{{}}}", &id)) // "{id}"
    }

}
