/*
 * Rayngin - 3D 6DF framework/engine for approach&click quests in rectangular chambers with objects consisting of balls
 * Copyright (c) 2021 Sunkware
 * PubKey FP: 6B6D C8E9 3438 6E9C 3D97  56E5 2CE9 A476 99EF 28F6
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * [ WWW: sunkware.org ]                         [ E-MAIL: sunkware@gmail.com ]
 */

extern crate serde;
extern crate serde_json;

use std::{
    fs,
    rc::Rc
};

use serde::Deserialize;

use crate::base::{
    BASE_DIR,
    JEXT,
    PrependErrorString
};

use crate::sys::{
    Font,
    FontContext,
    System
};

const FONTS_FILEPATH_NOEXT: &str = "fonts";

#[derive(Deserialize)]
struct FontDesc {
    file: String,
    size: i32
}

#[derive(Deserialize)]
struct FontsDesc {
    dir: String,
    null: FontDesc,
    meta: FontDesc,
    system: FontDesc,
    descript: FontDesc,
    inventory: FontDesc,
    hud_tempus: FontDesc,
    hud_reachable: FontDesc,
    talk_talker: FontDesc,
    talk_speech: FontDesc,
    talk_cue: FontDesc
}

#[derive(Clone, Copy, Hash, Eq, PartialEq, Debug)]
#[allow(non_camel_case_types)]
pub enum CFont {
    Meta, // this should be mono (constant width and height of chars) for Meta master to work properly
    System,
    Descript,
    Inventory,
    HUD_Tempus,
    HUD_Reachable,
    Talk_Talker,
    Talk_Speech,
    Talk_Cue
}

pub struct Fontset<'a> {
    null_font: Font<'a>,
    fonts: Vec<Font<'a>>
}

impl<'a> Fontset<'a> {
    pub fn load_fonts(ctx: &'a Rc<FontContext>) -> Result<Fontset<'a>, String> {
        let b = fs::read(format!("{}/{}.{}", BASE_DIR, FONTS_FILEPATH_NOEXT, JEXT)).pre_err("cannot read fonts file")?;
        let fsd: FontsDesc = serde_json::from_slice(&b).pre_err("cannot parse fonts json")?;

        let null_font = System::load_font(&ctx, format!("{}/{}", &fsd.dir, &fsd.null.file), fsd.null.size).pre_err("cannot load Null font")?;

        let mut fonts = Vec::<Font<'a>>::new();

        // Must be in the same order as CFont variants, for indexing to be correct
        fonts.push(System::load_font(&ctx, format!("{}/{}", &fsd.dir, &fsd.meta.file), fsd.meta.size).pre_err("cannot load Meta font")?);
        fonts.push(System::load_font(&ctx, format!("{}/{}", &fsd.dir, &fsd.system.file), fsd.system.size).pre_err("cannot load System font")?);
        fonts.push(System::load_font(&ctx, format!("{}/{}", &fsd.dir, &fsd.descript.file), fsd.descript.size).pre_err("cannot load Descript font")?);
        fonts.push(System::load_font(&ctx, format!("{}/{}", &fsd.dir, &fsd.inventory.file), fsd.inventory.size).pre_err("cannot load Inventory font")?);
        fonts.push(System::load_font(&ctx, format!("{}/{}", &fsd.dir, &fsd.hud_tempus.file), fsd.hud_tempus.size).pre_err("cannot load HUD Tempus font")?);
        fonts.push(System::load_font(&ctx, format!("{}/{}", &fsd.dir, &fsd.hud_reachable.file), fsd.hud_reachable.size).pre_err("cannot load HUD Reachable font")?);
        fonts.push(System::load_font(&ctx, format!("{}/{}", &fsd.dir, &fsd.talk_talker.file), fsd.talk_talker.size).pre_err("cannot load Talk Talker font")?);
        fonts.push(System::load_font(&ctx, format!("{}/{}", &fsd.dir, &fsd.talk_speech.file), fsd.talk_speech.size).pre_err("cannot load Talk Speech font")?);
        fonts.push(System::load_font(&ctx, format!("{}/{}", &fsd.dir, &fsd.talk_cue.file), fsd.talk_cue.size).pre_err("cannot load Talk Cue font")?);

        Ok(Fontset{null_font, fonts})
    }

    pub fn get(&self, cfont: CFont) -> &Font<'a> {
        let cfont = cfont as usize;
        if cfont < self.fonts.len() {
            &self.fonts[cfont]
        } else {
            &self.null_font
        }
    }

}
