/*
 * Rayngin - 3D 6DF framework/engine for approach&click quests in rectangular chambers with objects consisting of balls
 * Copyright (c) 2021 Sunkware
 * PubKey FP: 6B6D C8E9 3438 6E9C 3D97  56E5 2CE9 A476 99EF 28F6
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * [ WWW: sunkware.org ]                         [ E-MAIL: sunkware@gmail.com ]
 */

use std::{
    collections::HashMap,
    f64::consts::PI
};

use crate::base::{
    Attitude,
    Position
};

pub const MAX_BALL_RADIUS: i64 = 10000 << 0x10;
pub const MAX_ENTITY_SIZE: i64 = 100000 << 0x10;

#[derive(Clone, PartialEq, Debug)]
pub struct Ball {
    id: String,
    pub pos: Position,
    pub rad: i64,
    pub color: [u8; 4]
}

#[derive(Clone, PartialEq, Debug)]
pub struct Entity {
    id: String,
    pub att: Attitude,
    pub visible: bool,
    pub balls: Vec<Ball>,
    ball_hmap: HashMap<String, usize> // id to index
}

impl Ball {
    pub fn new(id: impl ToString, x: i64, y: i64, z: i64, rad: i64, color: [u8; 4]) -> Ball {
        let id = id.to_string();
        Ball{id, pos: Position::new(x << 0x10, y << 0x10, z << 0x10), rad: (rad << 0x10).min(MAX_BALL_RADIUS), color}
    }

    pub fn id(&self) -> &String {
        &(self.id)
    }
}

impl Entity {
    pub fn new(id: impl ToString, x: i64, y: i64, z: i64, yaw: f64, pitch: f64, roll: f64) -> Entity {
        let id = id.to_string();

        let visible = false;
        let balls: Vec<Ball> = Vec::new();
        let ball_hmap: HashMap<String, usize> = HashMap::new();

        Entity{
            id,
            att: Attitude::new(
                x << 0x10, y << 0x10, z << 0x10,
                yaw * PI / 180.0, pitch * PI / 180.0, roll * PI / 180.0
            ),
            visible, balls, ball_hmap
        }
    }

    pub fn dup(&self, id: impl ToString) -> Entity {
        let mut ent = self.clone();
        ent.id = id.to_string();
        ent
    }

    pub fn scale(&mut self, mult: f64) {
        let mult = (mult * 65536.0) as i64;
        for b in &mut self.balls {
            b.pos = b.pos.scaled(mult);
            b.rad = (b.rad * mult) >> 0x10;
        }
    }

    pub fn id(&self) -> &String {
        &(self.id)
    }

    pub fn add_ball(&mut self, ball: Ball) -> Result<(), String> {
        match self.ball_hmap.insert(ball.id.clone(), self.balls.len()) {
            Some(_) => Err(format!("entity '{}': cannot add ball '{}': already exists", self.id, ball.id)),
            None => {
                self.balls.push(ball);
                Ok(())
            }
        }
    }

    pub fn ball_ind(&self, id: impl ToString) -> Result<usize, String> {
        let id = id.to_string();
        match self.ball_hmap.get(&id) {
            Some(&i) => Ok(i),
            None => Err(format!("cannot find '{}' ball", &id))
        }
    }

}
