/*
 * Rayngin - 3D 6DF framework/engine for approach&click quests in rectangular chambers with objects consisting of balls
 * Copyright (c) 2021 Sunkware
 * PubKey FP: 6B6D C8E9 3438 6E9C 3D97  56E5 2CE9 A476 99EF 28F6
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * [ WWW: sunkware.org ]                         [ E-MAIL: sunkware@gmail.com ]
 */

extern crate serde;
extern crate serde_json;

use std::fs;

use serde::Deserialize;

use crate::base::{
    BASE_DIR,
	JEXT,
    PrependErrorString
};

use super::Cosmos;

use super::trajectory::{
    TimedAttitude,
    Trajectory
};

#[derive(Deserialize)]
struct TimedAttitudeDesc {
	time: f64,
	x: i64, y: i64, z: i64,
	yaw: f64, pitch: f64, roll: f64
}

#[derive(Deserialize)]
struct TrajectoryDesc {
    id: String,
    x: i64, y: i64, z: i64,
	timed_attitudes: Vec<TimedAttitudeDesc>,
	looped: bool
}


impl Cosmos {
	pub fn add_trajectory(&mut self, traj: Trajectory) -> Result<(), String> {
		match self.trajectory_hmap.insert(traj.id().clone(), self.trajectories.len()) {
			Some(_) => Err(format!("cannot add trajectory '{}': already exists", traj.id())),
			None => {
				self.trajectories.push(traj);
				Ok(())
			}
		}
	}

	pub fn load_trajectories(&mut self, base_filepath_noext: impl ToString) -> Result<(), String> {
		let filepath = format!("{}/{}.{}", BASE_DIR, base_filepath_noext.to_string(), JEXT);
		let b = fs::read(&filepath).pre_err("cannot read trajectories file")?;
		let tsd: Vec<TrajectoryDesc> = serde_json::from_slice(&b).pre_err("cannot parse trajectories json")?;

		for td in &tsd {
			let mut traj = Trajectory::new(td.id.clone(), td.x, td.y, td.z, td.looped);
			for tad in &td.timed_attitudes {
				let ta = TimedAttitude::new((tad.time * 1e6) as i128, tad.x, tad.y, tad.z, tad.yaw, tad.pitch, tad.roll);
				traj.add_timed_attitude(ta)?;
			}
			self.add_trajectory(traj)?;
		}

		Ok(())
	}

	pub fn trajectory_ind(&self, id: impl ToString) -> Result<usize, String> {
		let id = id.to_string();
		match self.trajectory_hmap.get(&id) {
			Some(&i) => Ok(i),
			None => Err(format!("cannot find '{}' trajectory", &id))
		}
	}

}
