/*
 * Rayngin - 3D 6DF framework/engine for approach&click quests in rectangular chambers with objects consisting of balls
 * Copyright (c) 2021 Sunkware
 * PubKey FP: 6B6D C8E9 3438 6E9C 3D97  56E5 2CE9 A476 99EF 28F6
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * [ WWW: sunkware.org ]                         [ E-MAIL: sunkware@gmail.com ]
 */

extern crate serde;
extern crate serde_json;

use std::fs;

use serde::Deserialize;

use crate::base::{
    BASE_DIR,
    JEXT,
    PrependErrorString
};

use super::Cosmos;

use super::texture::{
    SupraTexture,
    Texture
};

#[derive(Deserialize)]
struct TexturesDesc {
    dir: String,
    ext: String,
    ids: Vec<String>
}

#[derive(Deserialize)]
struct SupraTextureDesc {
    id: String,
    cell_size_binlog: u32,
    seed: u64,
    textures: Vec<String>
}

impl Cosmos {
    fn load_texture(&mut self, base_filepath: impl ToString, id: impl ToString) -> Result<(), String> {
        let base_filepath = base_filepath.to_string();
        let id = id.to_string();

        let texture = Texture::load(&base_filepath, &id).pre_err(format!("cannot load texture from {}", &base_filepath))?;
        match self.texture_hmap.insert(id.clone(), self.textures.len()) {
            Some(_) => Err(format!("texture '{}' already exists", &id)),
            None => {
                self.textures.push(texture);
                Ok(())
            }
        }
    }

    pub fn load_textures(&mut self, base_filepath_noext: impl ToString) -> Result<(), String> {
        let filepath = format!("{}/{}.{}", BASE_DIR, base_filepath_noext.to_string(), JEXT);
        let b = fs::read(&filepath).pre_err("cannot read textures file")?;
        let tsd: TexturesDesc = serde_json::from_slice(&b).pre_err("cannot parse textures json")?;

        for id in &tsd.ids {
            self.load_texture(format!("{}/{}.{}", &tsd.dir, id, &tsd.ext), id)?;
        }

        Ok(())
    }

    fn add_supra_texture(&mut self, id: impl ToString, cell_size_binlog: u32, seed: u64, txr_ids: &Vec<impl ToString>) -> Result<(), String> {
        let id = id.to_string();

        let mut txr = Vec::<usize>::new();
        for tid in txr_ids {
            match self.texture_hmap.get(&(tid.to_string())) {
                Some(v) => {
                    txr.push(*v);
                }
                None => {
                    return Err(format!("cannot add supra texture '{}': texture '{}' does not exist", id, tid.to_string()));
                }
            }
        }
        match self.supra_texture_hmap.insert(id.clone(), self.supra_texture_hmap.len()) {
            Some(_) => Err(format!("supra texture '{}' already exists", &id)),
            None => {
                self.supra_textures.push(SupraTexture::new(&id, cell_size_binlog, seed, &txr));
                Ok(())
            }
        }
    }

    pub fn load_supra_textures(&mut self, base_filepath_noext: impl ToString) -> Result<(), String> {
        let filepath = format!("{}/{}.{}", BASE_DIR, base_filepath_noext.to_string(), JEXT);
        let b = fs::read(&filepath).pre_err("cannot read supra textures file")?;
        let stsd: Vec<SupraTextureDesc> = serde_json::from_slice(&b).pre_err("cannot parse supra textures json")?;

        for std in &stsd {
            self.add_supra_texture(&std.id, std.cell_size_binlog, std.seed, &std.textures)?;
        }

        Ok(())
    }

}
