/*
 * Rayngin - 3D 6DF framework/engine for approach&click quests in rectangular chambers with objects consisting of balls
 * Copyright (c) 2021 Sunkware
 * PubKey FP: 6B6D C8E9 3438 6E9C 3D97  56E5 2CE9 A476 99EF 28F6
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * [ WWW: sunkware.org ]                         [ E-MAIL: sunkware@gmail.com ]
 */

extern crate serde;
extern crate serde_json;

use std::fs;

use serde::Deserialize;

use crate::base::{
    BASE_DIR,
    JEXT,
    Position,
    PrependErrorString
};

use super::Cosmos;

use super::chamber::Chamber;

#[derive(Deserialize)]
struct ChamberDesc {
    id: String,
    x: i64,
    y: i64,
    z: i64,
    lx: i64,
    ly: i64,
    lz: i64,
    supra_textures: [String; 6]
}

impl Cosmos {
    // DEBUG ?
    pub fn set_chamber_face_supra_texture(&mut self, ch_id: impl ToString, f: usize, sup_txr_id: impl ToString) -> Result<(), String> {
        let ch_id = ch_id.to_string();
        let sup_txr_id = sup_txr_id.to_string();

        match self.chamber_hmap.get(&ch_id) {
            Some(ind) => {
                if f < 6 {
                    match self.supra_texture_hmap.get(&sup_txr_id) {
                        Some(t) => {
                            self.chambers[*ind].sup_txr[f] = *t;
                            Ok(())
                        },
                        None => Err(format!("supra texture '{}' does not exist", sup_txr_id))
                    }
                } else {
                    Err(format!("face index {} too big", f))
                }
            },
            None => Err(format!("chamber '{}' does not exist", ch_id))
        }
    }

    // DEBUG ?
    pub fn set_chamber_supra_textures(&mut self, ch_id: impl ToString, sup_txr_ids: [impl ToString; 6]) -> Result<(), String> {
        let ch_id = ch_id.to_string();
        for f in 0..6 {
            let sup_txr_id = sup_txr_ids[f].to_string();
            self.set_chamber_face_supra_texture(ch_id.clone(), f, sup_txr_id)?;
        }
        Ok(())
    }

    pub fn verify_connect_chambers(&mut self) -> Result<usize, String> { // TODO: do it faster?
        let mut connected_pairs: usize = 0;

        if self.chambers.len() > 1 {
            for i in 0..(self.chambers.len()-1) {
                for j in (i+1)..self.chambers.len() {
                    match Chamber::connected(&self.chambers[i], &self.chambers[j]) {
                        Ok((b, face_i, face_j)) => {
                            if b {
                                self.chambers[i].adj[face_i].push(j);
                                self.chambers[j].adj[face_j].push(i);
                                connected_pairs += 1;
                            }
                        },
                        Err(s) => {
                            return Err(format!("error connecting chambers '{}' and '{}': {}", self.chambers[i].id(), self.chambers[j].id(), s));
                        }
                    }
                }
            }
        }

        Ok(connected_pairs)
    }

    pub fn add_chamber(&mut self, id: impl ToString, x: i64, y: i64, z: i64, lx: i64, ly: i64, lz: i64, sup_txr_ids: &[impl ToString; 6]) -> Result<(), String> {
        let id = id.to_string();

        let mut sup_txr: [usize; 6] = [0; 6];
        for i in 0..6 {
            match self.supra_texture_hmap.get(&sup_txr_ids[i].to_string()) {
                Some(v) => {
                    sup_txr[i] = *v;
                },
                None => {
                    return Err(format!("cannot add chamber '{}': supra texture '{}' does not exist", &id, &sup_txr_ids[i].to_string()));
                }
            }
        }

        match self.chamber_hmap.insert(id.clone(), self.chambers.len()) {
            Some(_) => Err(format!("cannot add chamber '{}': already exists", id)),
            None => {
                self.chambers.push(Chamber::new(id, x, y, z, lx, ly, lz, sup_txr));
                Ok(())
            }
        }
    }

    pub fn load_chambers(&mut self, base_filepath_noext: impl ToString) -> Result<(), String> {
        let filepath = format!("{}/{}.{}", BASE_DIR, base_filepath_noext.to_string(), JEXT);
        let b = fs::read(&filepath).pre_err("cannot read chambers file")?;
        let chsd: Vec<ChamberDesc> = serde_json::from_slice(&b).pre_err("cannot parse chambers json")?;

        for chd in &chsd {
            self.add_chamber(&chd.id, chd.x, chd.y, chd.z, chd.lx, chd.ly, chd.lz, &chd.supra_textures)?;
        }

        Ok(())
    }

    pub fn chamber_ind(&self, id: impl ToString) -> Result<usize, String> {
        let id = id.to_string();
        match self.chamber_hmap.get(&id) {
            Some(&i) => Ok(i),
            None => Err(format!("cannot find '{}' chamber", &id))
        }
    }

    #[inline(always)]
    pub fn chamber_by_pos(&self, pos: &Position) -> Option<usize> {
        for i in 0..self.chambers.len() {
            if self.chambers[i].inside(pos) {
                return Some(i)
            }
        }
        None
    }

}
