/*
 * Rayngin - 3D 6DF framework/engine for approach&click quests in rectangular chambers with objects consisting of balls
 * Copyright (c) 2021 Sunkware
 * PubKey FP: 6B6D C8E9 3438 6E9C 3D97  56E5 2CE9 A476 99EF 28F6
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * [ WWW: sunkware.org ]                         [ E-MAIL: sunkware@gmail.com ]
 */

use crate::base::Position;

/// Chamber is an (empty) axes-aligned 3D parallelepiped.
/// Chamber interiors should not intersect, only their faces may.
/// In right-hand Oxyz, x increases from back to front, y from left to right, z from bottom to top.
/// (x, y, z) is back-left-bottom corner. lx is length, ly is width, lz is height; these should be non-negative.
/// Faces are enumerated: back (0) and front (5), left (1) and right (4), bottom (2) and top (3); in each pair, sum is 5.
/// Each chamber keeps indices of chambers that intersect with its faces (ADJacent? ADJoined?).
/// Face i of the chamber intersects with face 5-i of the adjacent chamber.
/// This is to trace rays through chambers...
/// i-th face has supra texture with index sup_txr[i]
///
/// TODO: references instead of indices? something with Box/Rc/RefCell/...
pub struct Chamber {
    id: String,
    pub x: i64,
    pub y: i64,
    pub z: i64,
    pub lx: i64,
    pub ly: i64,
    pub lz: i64,
    pub sup_txr: [usize; 6],
    pub adj: [Vec<usize>; 6]
}
// Fixed point arithmetic, shift = 0x10

impl Chamber {
    pub fn new(id: impl ToString, mut x: i64, mut y: i64, mut z: i64, mut lx: i64, mut ly: i64, mut lz: i64, sup_txr: [usize; 6]) -> Chamber {
        let id = id.to_string();
        x <<= 0x10; y <<= 0x10; z <<= 0x10;
        lx <<= 0x10; ly <<= 0x10; lz <<= 0x10;
        let adj: [Vec<usize>; 6] = [Vec::new(), Vec::new(), Vec::new(), Vec::new(), Vec::new(), Vec::new()];
        Chamber{id, x, y, z, lx, ly, lz, sup_txr, adj}
    }

    pub fn id(&self) -> &String {
        &(self.id)
    }

    pub fn connected(a: &Chamber, b: &Chamber) -> Result<(bool, usize, usize), String> {
        // Verify that interiors do not intersect
        if ((a.x < b.x + b.lx) && (b.x < a.x + a.lx))
            && ((a.y < b.y + b.ly) && (b.y < a.y + a.ly))
            && ((a.z < b.z + b.lz) && (b.z < a.z + a.lz)) {
            return Err(format!("interiors of chambers intersect"));
        }

        let mut axis = "";
        let mut face_a: usize = 0;

        if a.x + a.lx == b.x { // face 5 of ch-i and face 0 of ch-j are coplanar
            axis = "x"; face_a = 5;
        } else if b.x + b.lx == a.x { // face 0 of ch-i and face 5 of ch-j are coplanar
            axis = "x"; face_a = 0;
        } else if a.y + a.ly == b.y { // face 4 of ch-i and face 1 of ch-j are coplanar
            axis = "y"; face_a = 4;
        } else if b.y + b.ly == a.y { // face 1 of ch-i and face 4 of ch-j are coplanar
            axis = "y"; face_a = 1;
        } else if a.z + a.lz == b.z { // face 3 of ch-i and face 2 of ch-j are coplanar
            axis = "z"; face_a = 3;
        } else if b.z + b.lz == a.z { // face 2 of ch-i and face 3 of ch-j are coplanar
            axis = "z"; face_a = 2;
        }

        let intersect = match axis {
            "x" => (a.y <= b.y + b.ly) && (b.y <= a.y + a.ly) && (a.z <= b.z + b.lz) && (b.z <= a.z + a.lz),
            "y" => (a.z <= b.z + b.lz) && (b.z <= a.z + a.lz) && (a.x <= b.x + b.lx) && (b.x <= a.x + a.lx),
            "z" => (a.x <= b.x + b.lx) && (b.x <= a.x + a.lx) && (a.y <= b.y + b.ly) && (b.y <= a.y + a.ly),
            _ => false
        };

        if intersect {
            Ok((true, face_a, 5 - face_a))
        } else {
            Ok((false, 6, 6))
        }
    }

    #[inline(always)]
    pub fn inside(&self, pos: &Position) -> bool {
        (pos.x >= self.x) && (pos.x <= self.x + self.lx) && (pos.y >= self.y) && (pos.y <= self.y + self.ly) && (pos.z >= self.z) && (pos.z <= self.z + self.lz)
    }
}
