/*
 * Rayngin - 3D 6DF framework/engine for approach&click quests in rectangular chambers with objects consisting of balls
 * Copyright (c) 2021 Sunkware
 * PubKey FP: 6B6D C8E9 3438 6E9C 3D97  56E5 2CE9 A476 99EF 28F6
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * [ WWW: sunkware.org ]                         [ E-MAIL: sunkware@gmail.com ]
 */

//! "Material": textures, chambers, entities, trajectores.
//! Fonts and localizations are here too.

use std::collections::HashMap;

use crate::base::{
	PrependErrorString,
	V3Di
};

mod chamber;
mod entity;
mod fontset;
mod lingua;
mod texture;
mod trajectory;

mod cosm_chamber;
mod cosm_entity;
mod cosm_texture;
mod cosm_trajectory;

pub use chamber::Chamber;

pub use entity::{
    MAX_ENTITY_SIZE,
    Ball,
    Entity
};

pub use fontset::{
    CFont,
    Fontset
};

pub use lingua::Lingua;

pub use texture::{
    SupraTexture,
    Texture
};

pub use trajectory::{
    TimedAttitude,
    Trajectory
};

const MAX_RAY_STEPS: i32 = 1 << 10; // in case ray tracing stucks...

pub struct Cosmos { // look, megalomania. "warning: struct is never constructed: `Cosmos`"
    pub textures: Vec<Texture>,
    texture_hmap: HashMap<String, usize>, // id to index

    pub supra_textures: Vec<SupraTexture>,
    supra_texture_hmap: HashMap<String, usize>, // id to index

    pub chambers: Vec<Chamber>,
    chamber_hmap: HashMap<String, usize>, // id to index

    pub entities: Vec<Entity>,
    entity_hmap: HashMap<String, usize>, // id to index

	pub trajectories: Vec<Trajectory>,
	trajectory_hmap: HashMap<String, usize> // id to index
}

impl Cosmos {
    pub fn new() -> Cosmos {
        let textures =  Vec::<Texture>::new();
        let texture_hmap =  HashMap::<String, usize>::new();

        let supra_textures = Vec::<SupraTexture>::new();
        let supra_texture_hmap = HashMap::<String, usize>::new();

        let chambers = Vec::<Chamber>::new();
        let chamber_hmap = HashMap::<String, usize>::new();

        let entities = Vec::<Entity>::new();
        let entity_hmap = HashMap::<String, usize>::new();

		let trajectories = Vec::<Trajectory>::new();
        let trajectory_hmap = HashMap::<String, usize>::new();

        Cosmos{
            textures, texture_hmap, supra_textures, supra_texture_hmap,
            chambers, chamber_hmap, entities, entity_hmap, trajectories, trajectory_hmap
        }
    }

	pub fn load(&mut self,
		textures_base_filepath_noext: impl ToString,
		supra_textures_base_filepath_noext: impl ToString,
		chambers_base_filepath_noext: impl ToString,
		entities_base_filepath_noext: impl ToString,
		trajectories_base_filepath_noext: impl ToString) -> Result<(), String> {

		let mut load = |load_func: fn(&mut Self, String) -> Result<(), String>, base_filepath_noext: String, res_name: String| -> Result<(), String> {
			match base_filepath_noext.is_empty() {
				false => load_func(self, base_filepath_noext.clone()).pre_err(format!("cannot load {} using '{}'", res_name, &base_filepath_noext)),
				true => Ok(())
			}
		};

		load(Self::load_textures, textures_base_filepath_noext.to_string(), String::from("textures"))?;
		load(Self::load_supra_textures, supra_textures_base_filepath_noext.to_string(), String::from("supra textures"))?;
		load(Self::load_chambers, chambers_base_filepath_noext.to_string(), String::from("chambers"))?;
		load(Self::load_entities, entities_base_filepath_noext.to_string(), String::from("entities"))?;
		load(Self::load_trajectories, trajectories_base_filepath_noext.to_string(), String::from("trajectories"))?;

		Ok(())
	}

	// See walls rendering, here's the same for given ray without obtaining pixel color, only distance
    pub fn ray_dist(&self, i_chamber: usize, point: &V3Di, ray: &V3Di) -> i64 {
        let mut dist: i64 = 0;

		let mut ch =  &(self.chambers[i_chamber]);

		let mut p = V3Di{x: point.x - ch.x, y: point.y - ch.y, z: point.z - ch.z};

        let ray_inv = V3Di{
			x: if ray.x != 0 { 0x1_0000_0000i64 / ray.x } else { 0 },
            y: if ray.y != 0 { 0x1_0000_0000i64 / ray.y } else { 0 },
            z: if ray.z != 0 { 0x1_0000_0000i64 / ray.z } else { 0 }
		};

		let mut ray_steps_left: i32 = MAX_RAY_STEPS; // in case ray tracing stucks

	    while ray_steps_left > 0 {
			ray_steps_left -= 1;

			let t = V3Di{
				x: match ray.x.signum() {
					-1 => (((ray_inv.x as i128) * ((- p.x) as i128)) >> 0x10) as i64,
					1 => (((ray_inv.x as i128) * ((ch.lx - p.x) as i128)) >> 0x10) as i64,
					_ => std::i64::MAX // "1/0 = ∞"
				},
				y: match ray.y.signum() {
					-1 => (((ray_inv.y as i128) * ((- p.y) as i128)) >> 0x10) as i64,
					1 => (((ray_inv.y as i128) * ((ch.ly - p.y) as i128)) >> 0x10) as i64,
					_ => std::i64::MAX // "1/0 = ∞"
				},
				z: match ray.z.signum() {
					-1 => (((ray_inv.z as i128) * ((- p.z) as i128)) >> 0x10) as i64,
					1 => (((ray_inv.z as i128) * ((ch.lz - p.z) as i128)) >> 0x10) as i64,
					_ => std::i64::MAX // "1/0 = ∞"
				}
			};

			let mut h = V3Di{x: 0, y: 0, z: 0}; // hit point (may be inaccurate...) in current chamber's reference frame (RF)

			let mut further: bool = false;

			// Minimal time and sign of ray component correspond to chamber face the ray actually hits
			// Check if there is an adjacent chamber at hit point,
			// recalculating hit point coordinates in adjacent chamber's RF
			// which are the coordinates of the point on traced ray at next iteration

			if t.z <= t.x {
				if t.z <= t.y { // t.z is min
					dist += t.z;
					h.x = p.x + ((((t.z as i128) * (ray.x as i128)) >> 0x10) as i64);
					h.y = p.y + ((((t.z as i128) * (ray.y as i128)) >> 0x10) as i64);
					let face = if ray.z > 0 { 3 } else { 2 };
					for i in &ch.adj[face] {
						let adj_ch = &(self.chambers[*i]);
						p.x = h.x + (ch.x - adj_ch.x);
						p.y = h.y + (ch.y - adj_ch.y);
						if (p.x >= 0) && (p.y >= 0) && (p.x <= adj_ch.lx) && (p.y <= adj_ch.ly) {
							p.z = if face == 2 { adj_ch.lz } else { 0 };
							ch = adj_ch;
							further = true;
							break;
						}
					}
				} else { // t.y is min
					dist += t.y;
					h.z = p.z + ((((t.y as i128) * (ray.z as i128)) >> 0x10) as i64);
					h.x = p.x + ((((t.y as i128) * (ray.x as i128)) >> 0x10) as i64);
					let face = if ray.y > 0 { 4 } else { 1 };
					for i in &ch.adj[face] {
						let adj_ch = &(self.chambers[*i]);
						p.x = h.x + (ch.x - adj_ch.x);
						p.z = h.z + (ch.z - adj_ch.z);
						if (p.x >= 0) && (p.z >= 0) && (p.x <= adj_ch.lx) && (p.z <= adj_ch.lz) {
							p.y = if face == 1 { adj_ch.ly } else { 0 };
							ch = adj_ch;
							further = true;
							break;
						}
					}
				}
			} else {
				if t.x <= t.y { // t.x is min
					dist += t.x;
					h.y = p.y + ((((t.x as i128) * (ray.y as i128)) >> 0x10) as i64);
					h.z = p.z + ((((t.x as i128) * (ray.z as i128)) >> 0x10) as i64);
					let face = if ray.x > 0 { 5 } else { 0 };
					for i in &ch.adj[face] {
						let adj_ch = &(self.chambers[*i]);
						p.y = h.y + (ch.y - adj_ch.y);
						p.z = h.z + (ch.z - adj_ch.z);
						if (p.y >= 0) && (p.z >= 0) && (p.y <= adj_ch.ly) && (p.z <= adj_ch.lz) {
							p.x = if face == 0 { adj_ch.lx } else { 0 };
							ch = adj_ch;
							further = true;
							break;
						}
					}
				} else { // t.y is min... duplicated
					dist += t.y;
					h.z = p.z + ((((t.y as i128) * (ray.z as i128)) >> 0x10) as i64);
					h.x = p.x + ((((t.y as i128) * (ray.x as i128)) >> 0x10) as i64);
					let face = if ray.y > 0 { 4 } else { 1 };
					for i in &ch.adj[face] {
						let adj_ch = &(self.chambers[*i]);
						p.x = h.x + (ch.x - adj_ch.x);
						p.z = h.z + (ch.z - adj_ch.z);
						if (p.x >= 0) && (p.z >= 0) && (p.x <= adj_ch.lx) && (p.z <= adj_ch.lz) {
							p.y = if face == 1 { adj_ch.ly } else { 0 };
							ch = adj_ch;
							further = true;
							break;
						}
					}
				}
			}

            if !further {
                break;
            }
        }

        dist
    }

}
