/*
 * Rayngin - 3D 6DF framework/engine for approach&click quests in rectangular chambers with objects consisting of balls
 * Copyright (c) 2021 Sunkware
 * PubKey FP: 6B6D C8E9 3438 6E9C 3D97  56E5 2CE9 A476 99EF 28F6
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * [ WWW: sunkware.org ]                         [ E-MAIL: sunkware@gmail.com ]
 */

//! Various low-level things:
//! error formatting, geometry-related structures and calculations, (A)RGB, ...
//! Some are in fixed point arithmetic and inlined.

extern crate serde;

use std::{
	collections::hash_map::DefaultHasher,
	f64::consts::PI,
	hash::{
		Hash,
		Hasher
	}
};

use serde::{
	Deserialize,
	Serialize
};

// ----------------------------------------------------------------

pub const BASE_DIR: &str = "_";
pub const SAVES_DIR: &str = "saves";

pub const JEXT: &str = "json";

pub const RAYNGIN_TITLE: &str = "Rayngin";
pub const RAYNGIN_COPYR: &str = "Copyright (c) 2021 Sunkware";
pub const RAYNGIN_URL: &str = "https://sunkware.org/rayngin";

pub const PLAY_TITLE: &str = "Walls and Balls"; // DEBUG
pub const PLAY_TITLE_EFFECT: &str = "W A L L S   A N D   B A L L S"; // DEBUG
pub const PLAY_COPYR: &str = "Copyright (c) ???? ????????";
pub const PLAY_URL: &str = "https://????????.???/???????";

pub const VERSION: &str = "0.3.2";

// ----------------------------------------------------------------

pub trait PrependErrorString<T> {
	fn pre_err(self: Self, p: impl ToString) -> Result<T, String>;
}

impl<T, E: ToString> PrependErrorString<T> for Result<T, E> {
	/// Ok() result is not changed, while
	/// error-as-string "s" is prepended with given string "p" and colon:
	/// p: s
	fn pre_err(self: Self, p: impl ToString) -> Result<T, String> {
		match self {
			Ok(t) => Ok(t),
			Err(e) => Err(format!("{}: {}", p.to_string(), e.to_string()))
		}
	}
}

// ----------------------------------------------------------------

pub trait Hash64 {
	fn hash64(self: &Self) -> u64;
}

impl<T: Hash> Hash64 for T {
	fn hash64(self: &Self) -> u64 {
		// from std::hash doc
		let mut hasher = DefaultHasher::new();
		self.hash(&mut hasher);
		hasher.finish()
	}
}

// ----------------------------------------------------------------

#[inline(always)]
pub fn binlog(mut n: i64) -> i32 {
    let mut bl: i32 = 0;
 	// Kind of binary search...
	if n > 0xFFFFFFFF {
		bl |= 0x20; n >>= 0x20;
	}
	if n > 0xFFFF {
		bl |= 0x10; n >>= 0x10;
	}
	if n > 0xFF {
		bl |= 8; n >>= 8;
	}
	if n > 0xF {
		bl |= 4; n >>= 4;
	}
	if n > 3 {
		bl |= 2; n >>= 2;
	}
	if n > 1 {
		bl |= 1;
	}
	bl
}

pub fn is_2pow(n: i64) -> bool { // is n = 2^m?
    (n & (n-1)) == 0
}

pub fn normalized_angle(mut a: f64) -> f64 { // to [-PI; PI)
	a = a.rem_euclid(2.0 * PI);
	if a >= PI {
		a -= 2.0 * PI;
	}
	a
}

#[inline(always)]
pub fn scale_u8(lum: u8, mult: i64, max_mult_binlog: i32) -> u8 {
	(((lum as i64) * mult) >> max_mult_binlog) as u8
}

// ----------------------------------------------------------------

#[derive(Clone, PartialEq, Debug)]
pub struct V3Df {
	pub x: f64,
	pub y: f64,
	pub z: f64
}

fn scalar_prod(a: &V3Df, b: &V3Df) -> f64 {
	a.x * b.x + a.y * b.y + a.z * b.z
}

fn vector_prod(a: &V3Df, b: &V3Df) -> V3Df {
	V3Df{
		x: a.y * b.z - a.z * b.y,
		y: a.z * b.x - a.x * b.z,
		z: a.x * b.y - a.y * b.x
	}
}

impl V3Df {
	pub fn is_zero(&self) -> bool {
		(self.x == 0.0) && (self.y == 0.0) && (self.z == 0.0)
	}

	pub fn norm_square(&self) -> f64 {
		self.x * self.x + self.y * self.y + self.z * self.z
	}

	pub fn norm(&self) -> f64 {
		(self.x * self.x + self.y * self.y + self.z * self.z).sqrt()
	}

	pub fn scaled(&self, c: f64) -> V3Df {
		V3Df{x: c * self.x, y: c * self.y, z: c * self.z}
	}

	pub fn normalized(&self) -> V3Df {
		if !self.is_zero() {
			self.scaled(1.0 / self.norm())
		} else {
			self.clone()
		}
	}

	// Rotated around axis by angle (counter-clockwise)
	pub fn rotated(&self, axis: &V3Df, angle: f64) -> V3Df {
		let e_a = axis.normalized();

		let d = if (e_a.x != 0.0) && (e_a.y == 0.0) {
			V3Df{x: 0.0, y: 1.0, z: 0.0}
		} else {
			V3Df{x: 1.0, y: 0.0, z: 0.0}
		}; // outside of Oe_a

		let c = scalar_prod(&d, &e_a); // coord of projection(d, Oe_a)
		let e_u = V3Df{x: d.x - c * e_a.x, y: d.y - c * e_a.y, z: d.z - c * e_a.z}.normalized(); // orthogonal to e_a

		let e_v = vector_prod(&e_a, &e_u); // (e_u, e_v, e_a) is a right-hand frame

 		// self in (e_u; e_v; e_a) frame
		let r = V3Df{x: scalar_prod(self, &e_u), y: scalar_prod(self, &e_v), z: scalar_prod(self, &e_a)};

		let (cos, sin) = (angle.cos(), angle.sin());

		// Rotation
		let r = V3Df{x: r.x * cos - r.y * sin, y: r.x * sin + r.y * cos, z: r.z};

		// Back to original RF
		V3Df{
			x: r.x * e_u.x + r.y * e_v.x + r.z * e_a.x,
			y: r.x * e_u.y + r.y * e_v.y + r.z * e_a.y,
			z: r.x * e_u.z + r.y * e_v.z + r.z * e_a.z
		}
	}

	pub fn rotated3(&self, axis1: &V3Df, angle1: f64, axis2: &V3Df, angle2: f64, axis3: &V3Df, angle3: f64) -> V3Df {
		self.rotated(axis1, angle1).rotated(axis2, angle2).rotated(axis3, angle3)
	}
}

// ----------------------------------------------------------------

#[derive(Clone, PartialEq, Debug, Default, Serialize, Deserialize)]
pub struct V3Di {
	pub x: i64,
	pub y: i64,
	pub z: i64
}

impl V3Di {
	pub fn new(x: i64, y: i64, z: i64) -> V3Di {
		V3Di{x, y, z}
	}

	#[inline(always)]
	pub fn shred(&self, shift: u32) -> V3Di {
		V3Di{x: self.x >> shift, y: self.y >> shift, z: self.z >> shift}
	}

	#[inline(always)]
	pub fn scaled(&self, c: i64) -> V3Di {
		V3Di{x: (self.x * c) >> 0x10, y: (self.y * c) >> 0x10, z: (self.z * c) >> 0x10}
	}

	#[inline(always)]
	pub fn scaled_noverflow(&self, c: i64) -> V3Di {
		let c = c as i128;
		V3Di{
			x: (((self.x as i128) * c) >> 0x10) as i64,
			y: (((self.y as i128) * c) >> 0x10) as i64,
			z: (((self.z as i128) * c) >> 0x10) as i64
		}
	}

	#[inline(always)]
	pub fn norm_square(&self) -> i128 {
		(self.x as i128) * (self.x as i128) + (self.y as i128) * (self.y as i128) + (self.z as i128) * (self.z as i128)
	}

	#[inline(always)]
	pub fn added(&self, other: &V3Di) -> V3Di {
		V3Di{x: self.x + other.x, y: self.y + other.y, z: self.z + other.z}
	}

	#[inline(always)]
	pub fn subbed(&self, other: &V3Di) -> V3Di {
		V3Di{x: self.x - other.x, y: self.y - other.y, z: self.z - other.z}
	}

	#[inline(always)]
	pub fn dist_square(&self, other: &Self) -> i128 {
		let (dx, dy, dz) = ((self.x - other.x) as i128, (self.y - other.y) as i128, (self.z - other.z) as i128);
		dx * dx + dy * dy + dz * dz
	}

	#[inline(always)]
	pub fn yawed(&self, cos: i64, sin: i64) -> V3Di { // around Oz
		V3Di{
			x: (self.x * cos - self.y * sin) >> 0x10,
			y: (self.x * sin + self.y * cos) >> 0x10,
			z: self.z
		}
	}

	#[inline(always)]
	pub fn yawed_noverflow(&self, cos: i64, sin: i64) -> V3Di { // around Oz
		let (cos, sin) = (cos as i128, sin as i128);
		V3Di{
			x: (((self.x as i128) * cos - (self.y as i128) * sin) >> 0x10) as i64,
			y: (((self.x as i128) * sin + (self.y as i128) * cos) >> 0x10) as i64,
			z: self.z
		}
	}

	#[inline(always)]
	pub fn pitched(&self, cos: i64, sin: i64) -> V3Di { // around Oy
		V3Di{
			x: (self.z * sin + self.x * cos) >> 0x10,
			y: self.y,
			z: (self.z * cos - self.x * sin) >> 0x10
		}
	}

	#[inline(always)]
	pub fn pitched_noverflow(&self, cos: i64, sin: i64) -> V3Di { // around Oy
		let (cos, sin) = (cos as i128, sin as i128);
		V3Di{
			x: (((self.z as i128) * sin + (self.x as i128) * cos) >> 0x10) as i64,
			y: self.y,
			z: (((self.z as i128) * cos - (self.x as i128) * sin) >> 0x10) as i64
		}
	}

	#[inline(always)]
	pub fn rolled(&self, cos: i64, sin: i64) -> V3Di { // around Ox
		V3Di{
			x: self.x,
			y: (self.y * cos - self.z * sin) >> 0x10,
			z: (self.y * sin + self.z * cos) >> 0x10
		}
	}

	#[inline(always)]
	pub fn rolled_noverflow(&self, cos: i64, sin: i64) -> V3Di { // around Ox
		let (cos, sin) = (cos as i128, sin as i128);
		V3Di{
			x: self.x,
			y: (((self.y as i128) * cos - (self.z as i128) * sin) >> 0x10) as i64,
			z: (((self.y as i128) * sin + (self.z as i128) * cos) >> 0x10) as i64
		}
	}

}

// ----------------------------------------------------------------

#[derive(Clone, PartialEq, Debug, Default, Serialize, Deserialize)]
pub struct Orientation {
	pub yaw: f64,
	pub pitch: f64,
	pub roll: f64
}

impl Orientation {
	pub fn new(yaw: f64, pitch: f64, roll: f64) -> Self {
		Self{yaw, pitch, roll}
	}

	pub fn cosines_sines_fpa(&self) -> (i64, i64, i64, i64, i64, i64) {
		(
			(self.yaw.cos() * 65536.0) as i64, (self.yaw.sin() * 65536.0) as i64,
			(self.pitch.cos() * 65536.0) as i64, (self.pitch.sin() * 65536.0) as i64,
			(self.roll.cos() * 65536.0) as i64, (self.roll.sin() * 65536.0) as i64
		)
	}

	pub fn from_reframe_axes(ox: &V3Df, oy: &V3Df, oz: &V3Df) -> Self {
	/*
		cy = cos(yaw), sy = sin(yaw), cp = cos(pitch), sp = sin(pitch), cr = cos(roll), sr = sin(roll)

		ex = (1, 0, 0), ey = (0, 1, 0), ez = (0, 0, 1)
		after roll:
		ex = (1, 0, 0), ey = (0, cr, sr), ez = (0, -sr, cr)
		after pitch:
		ex = (cp, 0, -sp), ey = (sr * sp, cr, sr * cp), ez = (cr * sp, -sr, cr * cp)
		after yaw:
		ex = (cp * cy, cp * sy, -sp) = (ox.x, ox.y, ox.z),
		ey = (sr * sp * cy - cr * sy, sr * sp * sy + cr * cy, sr * cp) = (oy.x, oy.y, oy.z),
		ez = (cr * sp * cy + sr * sy, cr * sp * sy - sr * cy, cr * cp) = (oz.x, oz.y, oz.z)

		6 unknowns, 9 equations

		-sp = ox.z => pitch = arcsin(-ox.z)
		if |pitch| < PI/2 <=> cp != 0, then
			(cy, sy) = (ox.x / cp, ox.y / cp)
			(cr, sr) = (oz.z / cp, oy.z / cp)
		else
			let yaw = 0, so cy = 1, sy = 0,
			(cr) = (oy.y), (-sr) = (oz.y)
			(
			 or	let roll = 0, so cr = 1, sr = 0,
				(-sy, cy) = (oy.x, oy.y)
			)
	*/
		let pitch = - ox.z.max(-1.0).min(1.0).asin();
		let cp = pitch.cos();
		let (yaw, roll) = if cp != 0.0 {
			( (ox.y / cp).atan2(ox.x / cp), (oy.z / cp).atan2(oz.z / cp) )
		} else {
			(0.0f64, -oz.y.atan2(oy.y) )
			// (- oy.x.atan2(oy.y), 0.0f64)
		};

		Self{yaw, pitch, roll}
	}

}

// ----------------------------------------------------------------

pub type Position = V3Di;

// ----------------------------------------------------------------

#[derive(Clone, PartialEq, Debug, Default, Serialize, Deserialize)]
pub struct Attitude {
	pub pos: Position,
	pub ori: Orientation
}

impl Attitude {
	pub fn new(x: i64, y: i64, z: i64, yaw: f64, pitch: f64, roll: f64) -> Self {
		Self{pos: Position::new(x, y, z), ori: Orientation::new(yaw, pitch, roll)}
	}

}

// ----------------------------------------------------------------

#[derive(Clone, Copy, PartialEq, Eq, Hash, Debug)]
pub struct RGB {
	pub r: u8,
	pub g: u8,
	pub b: u8
}

#[derive(Clone, Copy, PartialEq, Eq, Hash, Debug)]
pub struct ARGB {
	pub a: u8,
	pub r: u8,
	pub g: u8,
	pub b: u8
}
