use itertools::Itertools;
use lazy_static;
use regex::{Captures, Regex};

pub const WORDS_REGEX: &str = r"[a-z]+(?:[_\-][a-z]+)*";
pub const WORDS_UPPERCASE_REGEX: &str = r"[A-Z]+(?:[_\-][A-Z]+)*";

lazy_static! {
    static ref RE_INNER_PARENTHESIS: Regex = Regex::new(r"\(([^\(]+?)\)(\.\.\.)?").unwrap();
    static ref RE_INNER_BRACKETS: Regex = Regex::new(r"\[([^\[]+?)\](\.\.\.)?").unwrap();
    static ref RE_INNER_CURLY_BRACES: Regex = Regex::new(r"(\{[^\[]+?\})(\.\.\.)?").unwrap();
    static ref RE_REPEATABLE: Regex = Regex::new(&format!(
        r"(<{WORDS_REGEX}>|{WORDS_UPPERCASE_REGEX})\x20?(\.\.\.)"
    ))
    .unwrap();
}

#[derive(Debug, Clone, Copy)]
pub enum RegexMatch {
    InnerParenthesis,
    InnerBrackets,
    InnerCurlyBraces,
    Repeatable,
}

fn get_vec_from_cap(cap: &Captures) -> Vec<String> {
    cap.iter()
        .filter_map(|option_match| Some(option_match?.as_str().to_string()))
        .collect::<Vec<String>>()
}

pub fn expand_brackets(s: &str) -> String {
    let mut re_vec = RE_INNER_BRACKETS
        .captures_iter(s)
        .map(|cap| get_vec_from_cap(&cap))
        .filter(|cap| {
            cap[1]
                .split_whitespace()
                .filter(|x| !x.starts_with("<-"))
                .count()
                > 1
        })
        .collect::<Vec<_>>();
    re_vec.sort_by_key(|x| x[0].len());
    re_vec
        .iter()
        .find_map(|cap| {
            if cap[1].contains('|')
                || cap[1].ends_with("...")
                || RE_INNER_PARENTHESIS.captures(&cap[1]).is_some()
            {
                None
            } else {
                Some(expand_brackets(
                    &s.replacen(
                        &format!("[{}]", cap[1]),
                        &cap[1]
                            .split_whitespace()
                            .collect::<Vec<_>>()
                            .iter()
                            .circular_tuple_windows()
                            // <-o>, if - and then <, same group
                            .map(|(w, next)| {
                                if w.starts_with("<-") {
                                    format!("{}]", w)
                                } else if next.starts_with("<-") {
                                    format!("[{}", w)
                                } else {
                                    format!("[{}]", w)
                                }
                            })
                            .collect::<Vec<String>>()
                            .join(" "),
                        1,
                    ),
                ))
            }
        })
        .unwrap_or_else(|| s.to_string())
}

pub fn get_smallest_regex_match(usage: &str) -> Option<(RegexMatch, Vec<String>)> {
    let matcher = &[
        (
            RegexMatch::InnerParenthesis,
            RE_INNER_PARENTHESIS.captures(usage),
        ),
        (RegexMatch::InnerBrackets, RE_INNER_BRACKETS.captures(usage)),
        (
            RegexMatch::InnerCurlyBraces,
            RE_INNER_CURLY_BRACES.captures(usage),
        ),
        (RegexMatch::Repeatable, RE_REPEATABLE.captures(usage)),
    ];

    matcher
        .iter()
        .filter_map(|x| x.1.as_ref().map(|cap| (x.0, get_vec_from_cap(cap))))
        .min_by_key(|x| x.1[0].len())
}

pub fn split_keeping_separators(text: &str, split_chars: &[char]) -> Vec<String> {
    let mut result: Vec<String> = Vec::new();
    let mut last = 0;

    for (index, matched) in text.match_indices(|c: char| split_chars.contains(&c)) {
        if last != index {
            result.push(text[last..index].to_string());
        }
        result.push(matched.to_string());
        last = index + matched.len();
    }
    if last < text.len() {
        result.push(text[last..].to_string());
    }
    result
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_expand_brackets() {
        let usage = "foo [boo fuu] [-a -b -c] [zuu -d]".to_string();

        let result = expand_brackets(&usage);

        assert_eq!(
            result,
            "foo [boo] [fuu] [-a] [-b] [-c] [zuu] [-d]".to_string()
        )
    }

    #[test]
    fn test_expand_brackets_with_param() {
        let usage = "foo [boo fuu] [-o <-o> -a -b -c] [zuu -d]".to_string();

        let result = expand_brackets(&usage);

        assert_eq!(
            result,
            "foo [boo] [fuu] [-o <-o>] [-a] [-b] [-c] [zuu] [-d]".to_string()
        )
    }

    #[test]
    fn test_expand_brackets_with_param_and_white_spaces() {
        let usage = "foo [ boo fuu ] [ -o <-o> -a -b -c ] [ zuu -d ]".to_string();

        let result = expand_brackets(&usage);

        assert_eq!(
            result,
            "foo [boo] [fuu] [-o <-o>] [-a] [-b] [-c] [zuu] [-d]".to_string()
        )
    }

    #[test]
    fn test_expand_brackets_grouped() {
        let usage = "foo [(a | b)] [(a | b)]".to_string();

        let result = expand_brackets(&usage);

        assert_eq!(result, "foo [(a | b)] [(a | b)]".to_string())
    }

    #[test]
    fn test_expand_brackets_grouped_without_parentesis() {
        let usage = "foo [a | b] [a | b]".to_string();

        let result = expand_brackets(&usage);

        assert_eq!(result, "foo [a | b] [a | b]".to_string())
    }

    #[test]
    fn test_expand_brackets_multiple_groups() {
        let usage =
            "my_program.py [--help --sorted -o <-o>] [--quiet | --verbose] [INPUT ...]".to_string();

        let result = expand_brackets(&usage);

        assert_eq!(
            result,
            "my_program.py [--help] [--sorted] [-o <-o>] [--quiet | --verbose] [INPUT ...]"
                .to_string()
        )
    }

    #[test]
    fn test_split_keeping_separators() {
        let usage = "foo [boo fuu] [-o <-o> -a -b -c] [zuu -d]";

        let result: Vec<String> = usage
            .split_whitespace()
            .flat_map(|w| split_keeping_separators(&w, &['[', ']']))
            .collect();

        assert_eq!(
            result,
            vec![
                "foo".to_string(),
                "[".to_string(),
                "boo".to_string(),
                "fuu".to_string(),
                "]".to_string(),
                "[".to_string(),
                "-o".to_string(),
                "<-o>".to_string(),
                "-a".to_string(),
                "-b".to_string(),
                "-c".to_string(),
                "]".to_string(),
                "[".to_string(),
                "zuu".to_string(),
                "-d".to_string(),
                "]".to_string(),
            ]
        )
    }
}
