use std::{env, process};
use super::*;

#[derive(Debug, Clone)]
/// The main class for parsing
pub struct ArgParser {
    /// The name of the program
    name: String,
    /// The author of the program
    author: String,
    /// The version of the program
    version: String,
    /// Copyright string
    copyright: String,
    /// Simple description of the program
    info: String,
    /// Example usage of the program
    usage: String,
    /// Vec holding all the arguments
    args: Vec<Arg>,
    /// Whether the program *has* to have arguments passed or not
    require_args: bool,
    /// Any extra arguments that were passed, but not parsed
    pub extra: Vec<String>,
}

impl ArgParser {
    /// Creates a new ArgParser, with <name>. Creates help and version args by default
    pub fn new(name: &str) -> Self {
        let mut s = Self {
            name: String::from(name),
            author: String::new(),
            version: String::new(),
            copyright: String::new(),
            info: String::new(),
            usage: format!("{} [flags] [options]", name),
            args: Vec::new(),
            require_args: false,
            extra: Vec::new(),
        };

        s.args(
            vec!(
                Arg::bool("help", false)
                    .short('h')
                    .help("Prints this dialogue"),
                Arg::bool("version", false)
                    .short('v')
                    .help("Prints the version"),
            )
        );

        s
    }

    /// Sets the author
    pub fn author(&mut self, s: &str) -> &mut Self {
        self.author = String::from(s);
        self
    }

    /// Sets the version
    pub fn version(&mut self, s: &str) -> &mut Self {
        self.version = String::from(s);
        self
    }

    /// Sets the copyright
    pub fn copyright(&mut self, s: &str) -> &mut Self {
        self.copyright = String::from(s);
        self
    }

    /// Sets the info
    pub fn info(&mut self, s: &str) -> &mut Self {
        self.info = String::from(s);
        self
    }

    /// Sets the usage
    pub fn usage(&mut self, s: &str) -> &mut Self {
        self.usage = String::from(s);
        self
    }

    /// Sets whether the program needs arguments or not
    pub fn require_args(&mut self, b: bool) -> &mut Self {
        self.require_args = b;
        self
    }

    /// Adds the Vec<&mut Arg> to the parser's args
    pub fn args(&mut self, args: Vec<&mut Arg>) -> &mut Self{
        for arg in args {
            self.args.push(arg.clone());
        }
        self
    }

    fn set_arg(skip: &mut bool, arg: &mut Arg, next: Option<&String>) {
        if let ArgVal::Bool(b) = arg.val {
            arg.val = ArgVal::Bool(!b);
        } else {
            if let Some(next) = next {
                if next.starts_with("--") || next.starts_with('-') {
                    Self::error_exit(&format!("unexpected argument: {}", next));
                }
                arg.val = ArgVal::Str(next.to_owned());
                *skip = true;
            } else {
                    Self::error_exit(&format!("expected value for argument: {}", arg.name));
            }
        }
    }

    /// Gets a String value arg by <name>. If no arg is found, default to String::new()
    pub fn get_str(&self, name: &str) -> String {
        for arg in &self.args {
            if arg.name == name {
                if let ArgVal::Str(s) = &arg.val {
                    return s.clone();
                }
            }
        }
        String::new()
    }

    /// Gets a bool value arg by <name>. If no arg is found, default to false
    pub fn get_bool(&self, name: &str) -> bool {
        for arg in &self.args {
            if arg.name == name {
                if let ArgVal::Bool(b) = &arg.val {
                    return *b;
                }
            }
        }
        false
    }

    /// Parse with env::args()
    pub fn parse(&mut self) -> &mut Self {
        let mut args = env::args();
        // First argument will always be the path to the program itself. next() removes that.
        args.next();
        self.parse_vec(args.collect::<Vec<String>>())
    }

    /// Parses a given Vec<String>
    pub fn parse_vec(&mut self, args: Vec<String>) -> &mut Self {
        if args.len() == 0 && self.require_args {
            self.help_exit();
        }
        let mut skip = false;
        for (idx,arg) in args.iter().enumerate() {
            if skip {skip = false; continue}
            if let Some(arg) = arg.strip_prefix("--") {
                self.args.iter_mut()
                    .filter(|parg| parg.name == arg)
                    .for_each(|mut parg| Self::set_arg(&mut skip, &mut parg, args.get(idx+1)));
            } else if let Some(arg) = arg.strip_prefix('-') {
                for ch in arg.chars() {
                    self.args.iter_mut()
                        .filter(|parg| parg.short != None)
                        .filter(|parg| parg.short.unwrap() == ch)
                        .for_each(|mut parg| Self::set_arg(&mut skip, &mut parg, args.get(idx+1)));
                }
            } else {
                self.extra.push(arg.to_owned());
            }
        }
        if self.get_bool("help") {self.help_exit()}
        else if self.get_bool("version") {self.version_exit()}
        self
    }

    /// Prints help and exits
    fn help_exit(&self) {
        println!("{} {}\n{}\n{}\n{}", self.name, self.version, self.info, self.author, self.copyright);
        let mut flags = Vec::new();
        let mut options = Vec::new();
        for arg in &self.args {
            if let ArgVal::Bool(_) = arg.val {
                flags.push(arg);
            } else {
                options.push(arg);
            }
        }

        if flags.len() > 0 {
            println!("Flags:");
            for flag in flags {
                if let Some(short) = flag.short {
                    println!("\t-{}, --{}\t{}", short, flag.name, flag.help);
                } else {
                    println!("\t --{}\t{}", flag.name, flag.help);
                }
            }
        }

        if options.len() > 0 {
            println!("Options:");
            for option in options {
                if let Some(short) = option.short {
                    println!("\t-{}, --{}\t{}", short, option.name, option.help);
                } else {
                    println!("\t --{}\t{}", option.name, option.help);
                }
            }
        }
        process::exit(1);
    }

    /// Prints the version and exits
    fn version_exit(&self) {
        println!("{} {}", self.name, self.version);
        process::exit(1);
    }

    /// Prints error <message> and exits
    fn error_exit(message: &str) {
        eprintln!("Error: {}\nTry -h or --help for more information.", message);
        process::exit(1);
    }
}
