use crate::dynamics::RigidBodySet;
use crate::geometry::{ColliderSet, CollisionEvent, ContactPair};
use crossbeam::channel::Sender;

bitflags::bitflags! {
    #[cfg_attr(feature = "serde-serialize", derive(Serialize, Deserialize))]
    /// Flags affecting the events generated for this collider.
    pub struct ActiveEvents: u32 {
        /// If set, Rapier will call `EventHandler::handle_intersection_event` and
        /// `EventHandler::handle_contact_event` whenever relevant for this collider.
        const COLLISION_EVENTS = 0b0001;
    }
}

impl Default for ActiveEvents {
    fn default() -> Self {
        ActiveEvents::empty()
    }
}

/// Trait implemented by structures responsible for handling events generated by the physics engine.
///
/// Implementors of this trait will typically collect these events for future processing.
pub trait EventHandler: Send + Sync {
    /// Handle a collision event.
    ///
    /// A collision event is emitted when the state of intersection between two colliders changes.
    ///
    /// # Parameters
    /// * `event` - The collision event.
    /// * `bodies` - The set of rigid-bodies.
    /// * `colliders` - The set of colliders.
    /// * `contact_pair` - The current state of contacts between the two colliders. This is set ot `None`
    ///                    if at least one of the collider is a sensor (in which case no contact information
    ///                    is ever computed).
    fn handle_collision_event(
        &self,
        bodies: &RigidBodySet,
        colliders: &ColliderSet,
        event: CollisionEvent,
        contact_pair: Option<&ContactPair>,
    );
}

impl EventHandler for () {
    fn handle_collision_event(
        &self,
        _bodies: &RigidBodySet,
        _colliders: &ColliderSet,
        _event: CollisionEvent,
        _contact_pair: Option<&ContactPair>,
    ) {
    }
}

/// A collision event handler that collects events into a crossbeam channel.
pub struct ChannelEventCollector {
    event_sender: Sender<CollisionEvent>,
}

impl ChannelEventCollector {
    /// Initialize a new collision event handler from crossbeam channel senders.
    pub fn new(event_sender: Sender<CollisionEvent>) -> Self {
        Self { event_sender }
    }
}

impl EventHandler for ChannelEventCollector {
    fn handle_collision_event(
        &self,
        _bodies: &RigidBodySet,
        _colliders: &ColliderSet,
        event: CollisionEvent,
        _: Option<&ContactPair>,
    ) {
        let _ = self.event_sender.send(event);
    }
}
