//! Error handling with the `Result` type.
//!
//! [`Result<T, E>`][`Result`] is the type used for returning and propagating
//! errors. It is an enum with the variants, [`Ok(T)`], representing
//! success and containing a value, and [`Err(E)`], representing error
//! and containing an error value.
//!
//! ```
//! # #[allow(dead_code)]
//! enum Result<T, E> {
//!    Ok(T),
//!    Err(E),
//! }
//! ```
//!
//! Functions return [`Result`] whenever errors are expected and
//! recoverable. In the `std` crate, [`Result`] is most prominently used
//! for [I/O](../../std/io/index.html).
//!
//! A simple function returning [`Result`] might be
//! defined and used like so:
//!
//! ```
//! #[derive(Debug)]
//! enum Version { Version1, Version2 }
//!
//! fn parse_version(header: &[u8]) -> Result<Version, &'static str> {
//!     match header.get(0) {
//!         None => Err("invalid header length"),
//!         Some(&1) => Ok(Version::Version1),
//!         Some(&2) => Ok(Version::Version2),
//!         Some(_) => Err("invalid version"),
//!     }
//! }
//!
//! let version = parse_version(&[1, 2, 3, 4]);
//! match version {
//!     Ok(v) => println!("working with version: {:?}", v),
//!     Err(e) => println!("error parsing header: {:?}", e),
//! }
//! ```
//!
//! Pattern matching on [`Result`]s is clear and straightforward for
//! simple cases, but [`Result`] comes with some convenience methods
//! that make working with it more succinct.
//!
//! ```
//! let good_result: Result<i32, i32> = Ok(10);
//! let bad_result: Result<i32, i32> = Err(10);
//!
//! // The `is_ok` and `is_err` methods do what they say.
//! assert!(good_result.is_ok() && !good_result.is_err());
//! assert!(bad_result.is_err() && !bad_result.is_ok());
//!
//! // `map` consumes the `Result` and produces another.
//! let good_result: Result<i32, i32> = good_result.map(|i| i + 1);
//! let bad_result: Result<i32, i32> = bad_result.map(|i| i - 1);
//!
//! // Use `and_then` to continue the computation.
//! let good_result: Result<bool, i32> = good_result.and_then(|i| Ok(i == 11));
//!
//! // Use `or_else` to handle the error.
//! let bad_result: Result<i32, i32> = bad_result.or_else(|i| Ok(i + 20));
//!
//! // Consume the result and return the contents with `unwrap`.
//! let final_awesome_result = good_result.unwrap();
//! ```
//!
//! # Results must be used
//!
//! A common problem with using return values to indicate errors is
//! that it is easy to ignore the return value, thus failing to handle
//! the error. [`Result`] is annotated with the `#[must_use]` attribute,
//! which will cause the compiler to issue a warning when a Result
//! value is ignored. This makes [`Result`] especially useful with
//! functions that may encounter errors but don't otherwise return a
//! useful value.
//!
//! Consider the [`write_all`] method defined for I/O types
//! by the [`Write`] trait:
//!
//! ```
//! use std::io;
//!
//! trait Write {
//!     fn write_all(&mut self, bytes: &[u8]) -> Result<(), io::Error>;
//! }
//! ```
//!
//! *Note: The actual definition of [`Write`] uses [`io::Result`], which
//! is just a synonym for [`Result`]`<T, `[`io::Error`]`>`.*
//!
//! This method doesn't produce a value, but the write may
//! fail. It's crucial to handle the error case, and *not* write
//! something like this:
//!
//! ```no_run
//! # #![allow(unused_must_use)] // \o/
//! use std::fs::File;
//! use std::io::prelude::*;
//!
//! let mut file = File::create("valuable_data.txt").unwrap();
//! // If `write_all` errors, then we'll never know, because the return
//! // value is ignored.
//! file.write_all(b"important message");
//! ```
//!
//! If you *do* write that in Rust, the compiler will give you a
//! warning (by default, controlled by the `unused_must_use` lint).
//!
//! You might instead, if you don't want to handle the error, simply
//! assert success with [`expect`]. This will panic if the
//! write fails, providing a marginally useful message indicating why:
//!
//! ```no_run
//! use std::fs::File;
//! use std::io::prelude::*;
//!
//! let mut file = File::create("valuable_data.txt").unwrap();
//! file.write_all(b"important message").expect("failed to write message");
//! ```
//!
//! You might also simply assert success:
//!
//! ```no_run
//! # use std::fs::File;
//! # use std::io::prelude::*;
//! # let mut file = File::create("valuable_data.txt").unwrap();
//! assert!(file.write_all(b"important message").is_ok());
//! ```
//!
//! Or propagate the error up the call stack with [`?`]:
//!
//! ```
//! # use std::fs::File;
//! # use std::io::prelude::*;
//! # use std::io;
//! # #[allow(dead_code)]
//! fn write_message() -> io::Result<()> {
//!     let mut file = File::create("valuable_data.txt")?;
//!     file.write_all(b"important message")?;
//!     Ok(())
//! }
//! ```
//!
//! # The question mark operator, `?`
//!
//! When writing code that calls many functions that return the
//! [`Result`] type, the error handling can be tedious. The question mark
//! operator, [`?`], hides some of the boilerplate of propagating errors
//! up the call stack.
//!
//! It replaces this:
//!
//! ```
//! # #![allow(dead_code)]
//! use std::fs::File;
//! use std::io::prelude::*;
//! use std::io;
//!
//! struct Info {
//!     name: String,
//!     age: i32,
//!     rating: i32,
//! }
//!
//! fn write_info(info: &Info) -> io::Result<()> {
//!     // Early return on error
//!     let mut file = match File::create("my_best_friends.txt") {
//!            Err(e) => return Err(e),
//!            Ok(f) => f,
//!     };
//!     if let Err(e) = file.write_all(format!("name: {}\n", info.name).as_bytes()) {
//!         return Err(e)
//!     }
//!     if let Err(e) = file.write_all(format!("age: {}\n", info.age).as_bytes()) {
//!         return Err(e)
//!     }
//!     if let Err(e) = file.write_all(format!("rating: {}\n", info.rating).as_bytes()) {
//!         return Err(e)
//!     }
//!     Ok(())
//! }
//! ```
//!
//! With this:
//!
//! ```
//! # #![allow(dead_code)]
//! use std::fs::File;
//! use std::io::prelude::*;
//! use std::io;
//!
//! struct Info {
//!     name: String,
//!     age: i32,
//!     rating: i32,
//! }
//!
//! fn write_info(info: &Info) -> io::Result<()> {
//!     let mut file = File::create("my_best_friends.txt")?;
//!     // Early return on error
//!     file.write_all(format!("name: {}\n", info.name).as_bytes())?;
//!     file.write_all(format!("age: {}\n", info.age).as_bytes())?;
//!     file.write_all(format!("rating: {}\n", info.rating).as_bytes())?;
//!     Ok(())
//! }
//! ```
//!
//! *It's much nicer!*
//!
//! Ending the expression with [`?`] will result in the unwrapped
//! success ([`Ok`]) value, unless the result is [`Err`], in which case
//! [`Err`] is returned early from the enclosing function.
//!
//! [`?`] can only be used in functions that return [`Result`] because of the
//! early return of [`Err`] that it provides.
//!
//! [`expect`]: Result::expect
//! [`Write`]: ../../std/io/trait.Write.html
//! [`write_all`]: ../../std/io/trait.Write.html#method.write_all
//! [`io::Result`]: ../../std/io/type.Result.html
//! [`?`]: crate::ops::Try
//! [`Ok(T)`]: Ok
//! [`Err(E)`]: Err
//! [`io::Error`]: ../../std/io/struct.Error.html

use std::future::Future;
use std::hint;

/////////////////////////////////////////////////////////////////////////////
// Type implementation
/////////////////////////////////////////////////////////////////////////////

pub trait BlockInPlaceResult<T, E> {
    /////////////////////////////////////////////////////////////////////////
    // Adapter for each variant
    /////////////////////////////////////////////////////////////////////////

    /// Converts from `Result<T, E>` to [`Option<T>`].
    ///
    /// Converts `self` into an [`Option<T>`], consuming `self`,
    /// and discarding the error, if any.
    ///
    /// # Examples
    ///
    /// Basic usage:
    ///
    /// ```
    /// let x: Result<u32, &str> = Ok(2);
    /// assert_eq!(x.ok(), Some(2));
    ///
    /// let x: Result<u32, &str> = Err("Nothing here");
    /// assert_eq!(x.ok(), None);
    /// ```
    fn ok(self) -> Option<T>;

    /// Converts from `Result<T, E>` to [`Option<E>`].
    ///
    /// Converts `self` into an [`Option<E>`], consuming `self`,
    /// and discarding the success value, if any.
    ///
    /// # Examples
    ///
    /// Basic usage:
    ///
    /// ```
    /// let x: Result<u32, &str> = Ok(2);
    /// assert_eq!(x.err(), None);
    ///
    /// let x: Result<u32, &str> = Err("Nothing here");
    /// assert_eq!(x.err(), Some("Nothing here"));
    /// ```
    fn err(self) -> Option<E>;

    /////////////////////////////////////////////////////////////////////////
    // Transforming contained values
    /////////////////////////////////////////////////////////////////////////

    /// Maps a `Result<T, E>` to `Result<U, E>` by applying a function to a
    /// contained [`Ok`] value, leaving an [`Err`] value untouched.
    ///
    /// This function can be used to compose the results of two functions.
    ///
    /// # Examples
    ///
    /// Print the numbers on each line of a string multiplied by two.
    ///
    /// ```
    /// let line = "1\n2\n3\n4\n";
    ///
    /// for num in line.lines() {
    ///     match num.parse::<i32>().map(|i| i * 2) {
    ///         Ok(n) => println!("{}", n),
    ///         Err(..) => {}
    ///     }
    /// }
    /// ```
    fn map<U, F: FnOnce(T) -> U>(self, op: F) -> Result<U, E>;

    /// Applies a function to the contained value (if [`Ok`]),
    /// or returns the provided default (if [`Err`]).
    ///
    /// Arguments passed to `map_or` are eagerly evaluated; if you are passing
    /// the result of a function call, it is recommended to use [`map_or_else`],
    /// which is lazily evaluated.
    ///
    /// [`map_or_else`]: Result::map_or_else
    ///
    /// # Examples
    ///
    /// ```
    /// let x: Result<_, &str> = Ok("foo");
    /// assert_eq!(x.map_or(42, |v| v.len()), 3);
    ///
    /// let x: Result<&str, _> = Err("bar");
    /// assert_eq!(x.map_or(42, |v| v.len()), 42);
    /// ```
    fn map_or<U, F: FnOnce(T) -> U>(self, default: U, f: F) -> U;

    /// Maps a `Result<T, E>` to `U` by applying a function to a
    /// contained [`Ok`] value, or a fallback function to a
    /// contained [`Err`] value.
    ///
    /// This function can be used to unpack a successful result
    /// while handling an error.
    ///
    ///
    /// # Examples
    ///
    /// Basic usage:
    ///
    /// ```
    /// let k = 21;
    ///
    /// let x : Result<_, &str> = Ok("foo");
    /// assert_eq!(x.map_or_else(|e| k * 2, |v| v.len()), 3);
    ///
    /// let x : Result<&str, _> = Err("bar");
    /// assert_eq!(x.map_or_else(|e| k * 2, |v| v.len()), 42);
    /// ```
    fn map_or_else<U, D: FnOnce(E) -> U, F: FnOnce(T) -> U>(self, default: D, f: F) -> U;

    /// Maps a `Result<T, E>` to `Result<T, F>` by applying a function to a
    /// contained [`Err`] value, leaving an [`Ok`] value untouched.
    ///
    /// This function can be used to pass through a successful result while handling
    /// an error.
    ///
    ///
    /// # Examples
    ///
    /// Basic usage:
    ///
    /// ```
    /// fn stringify(x: u32) -> String { format!("error code: {}", x) }
    ///
    /// let x: Result<u32, u32> = Ok(2);
    /// assert_eq!(x.map_err(stringify), Ok(2));
    ///
    /// let x: Result<u32, u32> = Err(13);
    /// assert_eq!(x.map_err(stringify), Err("error code: 13".to_string()));
    /// ```
    fn map_err<F, O: FnOnce(E) -> F>(self, op: O) -> Result<T, F>;

    ////////////////////////////////////////////////////////////////////////
    // Boolean operations on the values, eager and lazy
    /////////////////////////////////////////////////////////////////////////

    /// Returns `res` if the result is [`Ok`], otherwise returns the [`Err`] value of `self`.
    ///
    ///
    /// # Examples
    ///
    /// Basic usage:
    ///
    /// ```
    /// let x: Result<u32, &str> = Ok(2);
    /// let y: Result<&str, &str> = Err("late error");
    /// assert_eq!(x.and(y), Err("late error"));
    ///
    /// let x: Result<u32, &str> = Err("early error");
    /// let y: Result<&str, &str> = Ok("foo");
    /// assert_eq!(x.and(y), Err("early error"));
    ///
    /// let x: Result<u32, &str> = Err("not a 2");
    /// let y: Result<&str, &str> = Err("late error");
    /// assert_eq!(x.and(y), Err("not a 2"));
    ///
    /// let x: Result<u32, &str> = Ok(2);
    /// let y: Result<&str, &str> = Ok("different result type");
    /// assert_eq!(x.and(y), Ok("different result type"));
    /// ```
    fn and<U>(self, res: Result<U, E>) -> Result<U, E>;

    /// Calls `op` if the result is [`Ok`], otherwise returns the [`Err`] value of `self`.
    ///
    ///
    /// This function can be used for control flow based on `Result` values.
    ///
    /// # Examples
    ///
    /// Basic usage:
    ///
    /// ```
    /// fn sq(x: u32) -> Result<u32, u32> { Ok(x * x) }
    /// fn err(x: u32) -> Result<u32, u32> { Err(x) }
    ///
    /// assert_eq!(Ok(2).and_then(sq).and_then(sq), Ok(16));
    /// assert_eq!(Ok(2).and_then(sq).and_then(err), Err(4));
    /// assert_eq!(Ok(2).and_then(err).and_then(sq), Err(2));
    /// assert_eq!(Err(3).and_then(sq).and_then(sq), Err(3));
    /// ```
    fn and_then<U, F: FnOnce(T) -> Result<U, E>>(self, op: F) -> Result<U, E>;

    /// Returns `res` if the result is [`Err`], otherwise returns the [`Ok`] value of `self`.
    ///
    /// Arguments passed to `or` are eagerly evaluated; if you are passing the
    /// result of a function call, it is recommended to use [`or_else`], which is
    /// lazily evaluated.
    ///
    /// [`or_else`]: Result::or_else
    ///
    /// # Examples
    ///
    /// Basic usage:
    ///
    /// ```
    /// let x: Result<u32, &str> = Ok(2);
    /// let y: Result<u32, &str> = Err("late error");
    /// assert_eq!(x.or(y), Ok(2));
    ///
    /// let x: Result<u32, &str> = Err("early error");
    /// let y: Result<u32, &str> = Ok(2);
    /// assert_eq!(x.or(y), Ok(2));
    ///
    /// let x: Result<u32, &str> = Err("not a 2");
    /// let y: Result<u32, &str> = Err("late error");
    /// assert_eq!(x.or(y), Err("late error"));
    ///
    /// let x: Result<u32, &str> = Ok(2);
    /// let y: Result<u32, &str> = Ok(100);
    /// assert_eq!(x.or(y), Ok(2));
    /// ```
    fn or<F>(self, res: Result<T, F>) -> Result<T, F>;

    /// Calls `op` if the result is [`Err`], otherwise returns the [`Ok`] value of `self`.
    ///
    /// This function can be used for control flow based on result values.
    ///
    ///
    /// # Examples
    ///
    /// Basic usage:
    ///
    /// ```
    /// fn sq(x: u32) -> Result<u32, u32> { Ok(x * x) }
    /// fn err(x: u32) -> Result<u32, u32> { Err(x) }
    ///
    /// assert_eq!(Ok(2).or_else(sq).or_else(sq), Ok(2));
    /// assert_eq!(Ok(2).or_else(err).or_else(sq), Ok(2));
    /// assert_eq!(Err(3).or_else(sq).or_else(err), Ok(9));
    /// assert_eq!(Err(3).or_else(err).or_else(err), Err(3));
    /// ```
    fn or_else<F, O: FnOnce(E) -> Result<T, F>>(self, op: O) -> Result<T, F>;

    /// Returns the contained [`Ok`] value or a provided default.
    ///
    /// Arguments passed to `unwrap_or` are eagerly evaluated; if you are passing
    /// the result of a function call, it is recommended to use [`unwrap_or_else`],
    /// which is lazily evaluated.
    ///
    /// [`unwrap_or_else`]: Result::unwrap_or_else
    ///
    /// # Examples
    ///
    /// Basic usage:
    ///
    /// ```
    /// let default = 2;
    /// let x: Result<u32, &str> = Ok(9);
    /// assert_eq!(x.unwrap_or(default), 9);
    ///
    /// let x: Result<u32, &str> = Err("error");
    /// assert_eq!(x.unwrap_or(default), default);
    /// ```
    fn unwrap_or(self, default: T) -> T;

    /// Returns the contained [`Ok`] value or computes it from a closure.
    ///
    ///
    /// # Examples
    ///
    /// Basic usage:
    ///
    /// ```
    /// fn count(x: &str) -> usize { x.len() }
    ///
    /// assert_eq!(Ok(2).unwrap_or_else(count), 2);
    /// assert_eq!(Err("foo").unwrap_or_else(count), 3);
    /// ```
    fn unwrap_or_else<F: FnOnce(E) -> T>(self, op: F) -> T;

    /// Returns the contained [`Ok`] value, consuming the `self` value,
    /// without checking that the value is not an [`Err`].
    ///
    /// # Safety
    ///
    /// Calling this method on an [`Err`] is *[undefined behavior]*.
    ///
    /// [undefined behavior]: https://doc.rust-lang.org/reference/behavior-considered-undefined.html
    ///
    /// # Examples
    ///
    /// ```
    /// use railsgun::BlockInPlaceResult;
    /// # use tokio;
    /// # tokio::runtime::Builder::new_multi_thread()
    /// #      .worker_threads(1)
    /// #      .thread_name("rocket-worker-thread")
    /// #      .enable_all()
    /// #      .build()
    /// #      .expect("create tokio runtime")
    /// #      .block_on(async {
    ///
    /// assert_eq!(unsafe { async {Ok(2) as Result<_, &str>}.unwrap_unchecked() }, 2);
    /// # });
    /// ```
    ///
    /// ```no_run
    /// use railsgun::BlockInPlaceResult;
    /// # use tokio;
    /// # tokio::runtime::Builder::new_multi_thread()
    /// #      .worker_threads(1)
    /// #      .thread_name("rocket-worker-thread")
    /// #      .enable_all()
    /// #      .build()
    /// #      .expect("create tokio runtime")
    /// #      .block_on(async {
    ///
    /// unsafe { async {Err("emergency failure") as Result<u32, _>}.unwrap_unchecked(); } // Undefined behavior!
    /// # });
    /// ```
    #[track_caller]
    unsafe fn unwrap_unchecked(self) -> T;

    /// Returns the contained [`Err`] value, consuming the `self` value,
    /// without checking that the value is not an [`Ok`].
    ///
    /// # Safety
    ///
    /// Calling this method on an [`Ok`] is *[undefined behavior]*.
    ///
    /// [undefined behavior]: https://doc.rust-lang.org/reference/behavior-considered-undefined.html
    ///
    /// # Examples
    ///
    /// ```no_run
    /// use railsgun::BlockInPlaceResult;
    /// # use tokio;
    /// # tokio::runtime::Builder::new_multi_thread()
    /// #      .worker_threads(1)
    /// #      .thread_name("rocket-worker-thread")
    /// #      .enable_all()
    /// #      .build()
    /// #      .expect("create tokio runtime")
    /// #      .block_on(async {
    ///
    /// unsafe { async {Ok(2) as Result<_, &str>}.unwrap_err_unchecked() }; // Undefined behavior!
    /// #      });
    /// ```
    ///
    /// ```
    /// use railsgun::BlockInPlaceResult;
    /// # use tokio;
    /// # tokio::runtime::Builder::new_multi_thread()
    /// #      .worker_threads(1)
    /// #      .thread_name("rocket-worker-thread")
    /// #      .enable_all()
    /// #      .build()
    /// #      .expect("create tokio runtime")
    /// #      .block_on(async {
    /// assert_eq!(unsafe { async {Err("emergency failure") as Result<u32, _>}.unwrap_err_unchecked() }, "emergency failure");
    /// #      });
    /// ```
    #[track_caller]
    unsafe fn unwrap_err_unchecked(self) -> E;
}

impl<T, E, L> BlockInPlaceResult<T, E> for L
where
    L: Future<Output = Result<T, E>>,
{
    /////////////////////////////////////////////////////////////////////////
    // Adapter for each variant
    /////////////////////////////////////////////////////////////////////////

    /// Converts from `Result<T, E>` to [`Option<T>`].
    ///
    /// Converts `self` into an [`Option<T>`], consuming `self`,
    /// and discarding the error, if any.
    ///
    /// # Examples
    ///
    /// Basic usage:
    ///
    /// ```
    /// use railsgun::BlockInPlaceResult;
    ///
    /// # async fn run() -> () {
    /// let x = async {Ok(2u32) as Result<u32, &str>};
    /// assert_eq!(x.ok(), Some(2));
    ///
    /// let x = async {Err("Nothing here") as Result<u32,_>};
    /// assert_eq!(x.ok(), None);
    /// # }
    /// ```
    #[inline]
    fn ok(self) -> Option<T> {
        async_handle(async {
            match self.await {
                Ok(x) => Some(x),
                Err(_) => None,
            }
        })
    }

    /// Converts from `Result<T, E>` to [`Option<E>`].
    ///
    /// Converts `self` into an [`Option<E>`], consuming `self`,
    /// and discarding the success value, if any.
    ///
    /// # Examples
    ///
    /// Basic usage:
    ///
    /// ```
    /// use railsgun::BlockInPlaceResult;
    ///
    /// # async fn run() -> () {
    /// let x = async {Ok(2u32) as Result<u32, &str>};
    /// assert_eq!(x.err(), None);
    ///
    /// let x = async {Err("Nothing here") as Result<u32,&str>};
    /// assert_eq!(x.err(), Some("Nothing here"));
    /// }
    /// ```
    #[inline]
    fn err(self) -> Option<E> {
        async_handle(async {
            match self.await {
                Ok(_) => None,
                Err(x) => Some(x),
            }
        })
    }

    /////////////////////////////////////////////////////////////////////////
    // Transforming contained values
    /////////////////////////////////////////////////////////////////////////

    /// Maps a `Result<T, E>` to `Result<U, E>` by applying a function to a
    /// contained [`Ok`] value, leaving an [`Err`] value untouched.
    ///
    /// This function can be used to compose the results of two functions.
    ///
    /// # Examples
    ///
    /// Print the numbers on each line of a string multiplied by two.
    ///
    /// ```
    /// use railsgun::BlockInPlaceResult;
    /// async fn async_function() -> Result<String, String> {
    ///    Ok("foo".to_string())
    /// }
    ///
    /// # async fn run() -> () {
    /// async_function()
    ///     .map(|t| { format!("Magic: {}", t) })
    ///     .map(|t| { println!("{}", t) })
    ///     .ok();
    /// # }
    /// ```
    #[inline]
    fn map<U, F: FnOnce(T) -> U>(self, op: F) -> Result<U, E> {
        async_handle(async {
            match self.await {
                Ok(t) => Ok(op(t)),
                Err(e) => Err(e),
            }
        })
    }

    /// Applies a function to the contained value (if [`Ok`]),
    /// or returns the provided default (if [`Err`]).
    ///
    /// Arguments passed to `map_or` are eagerly evaluated; if you are passing
    /// the result of a function call, it is recommended to use [`map_or_else`],
    /// which is lazily evaluated.
    ///
    /// [`map_or_else`]: Result::map_or_else
    ///
    /// # Examples
    ///
    /// ```
    /// use railsgun::BlockInPlaceResult;
    ///
    /// # async fn run() -> () {
    /// let x = async {Ok("foo") as Result<&str, &str>};
    /// assert_eq!(x.map_or(42, |v| v.len()), 3);
    ///
    /// let x = async {Err("bar") as Result<&str, &str>};
    /// assert_eq!(x.map_or(42, |v| v.len()), 42);
    /// # }
    /// ```
    #[inline]
    fn map_or<U, F: FnOnce(T) -> U>(self, default: U, f: F) -> U {
        async_handle(async {
            match self.await {
                Ok(t) => f(t),
                Err(_) => default,
            }
        })
    }

    /// Maps a `Result<T, E>` to `U` by applying a function to a
    /// contained [`Ok`] value, or a fallback function to a
    /// contained [`Err`] value.
    ///
    /// This function can be used to unpack a successful result
    /// while handling an error.
    ///
    ///
    /// # Examples
    ///
    /// Basic usage:
    ///
    /// ```
    /// use railsgun::BlockInPlaceResult;
    ///
    /// # async fn run() -> () {
    /// let k = 21;
    ///
    /// let x = async {Ok("foo") as Result<&str,&str>};
    /// assert_eq!(x.map_or_else(|e| k * 2, |v| v.len()), 3);
    ///
    /// let x = async {Err("bar") as Result<&str,&str>};
    /// assert_eq!(x.map_or_else(|e| k * 2, |v| v.len()), 42);
    /// # }
    /// ```
    #[inline]
    fn map_or_else<U, D: FnOnce(E) -> U, F: FnOnce(T) -> U>(self, default: D, f: F) -> U {
        async_handle(async {
            match self.await {
                Ok(t) => f(t),
                Err(e) => default(e),
            }
        })
    }

    /// Maps a `Result<T, E>` to `Result<T, F>` by applying a function to a
    /// contained [`Err`] value, leaving an [`Ok`] value untouched.
    ///
    /// This function can be used to pass through a successful result while handling
    /// an error.
    ///
    ///
    /// # Examples
    ///
    /// Basic usage:
    ///
    /// ```
    /// use railsgun::BlockInPlaceResult;
    ///
    /// # async fn run() -> () {
    /// fn stringify(x: u32) -> String { format!("error code: {}", x) }
    ///
    /// let x = async {Ok(2u32) as Result<u32,u32>};
    /// assert_eq!(x.map_err(stringify), Ok(2));
    ///
    /// let x = async {Err(13u32) as Result<u32,u32>};
    /// assert_eq!(x.map_err(stringify), Err("error code: 13".to_string()));
    /// # }
    /// ```
    #[inline]
    fn map_err<F, O: FnOnce(E) -> F>(self, op: O) -> Result<T, F> {
        async_handle(async {
            match self.await {
                Ok(t) => Ok(t),
                Err(e) => Err(op(e)),
            }
        })
    }

    ////////////////////////////////////////////////////////////////////////
    // Boolean operations on the values, eager and lazy
    /////////////////////////////////////////////////////////////////////////

    /// Returns `res` if the result is [`Ok`], otherwise returns the [`Err`] value of `self`.
    ///
    ///
    /// # Examples
    ///
    /// Basic usage:
    ///
    /// ```
    /// use railsgun::BlockInPlaceResult;
    ///
    /// # async fn run() -> () {
    /// let x = async { Ok(2u32) as Result<u32, &str>};
    /// let y: Result<&str, &str> = Err("late error");
    /// assert_eq!(x.and(y), Err("late error"));
    ///
    /// let x = async {  Err("early error") as Result<u32, &str>};
    /// let y: Result<&str, &str> = Ok("foo");
    /// assert_eq!(x.and(y), Err("early error"));
    ///
    /// let x = async {  Err("not a 2") as Result<u32, &str>};
    /// let y: Result<&str, &str> = Err("late error");
    /// assert_eq!(x.and(y), Err("not a 2"));
    ///
    /// let x = async {  Ok(2u32) as Result<u32, &str>};
    /// let y: Result<&str, &str> = Ok("different result type");
    /// assert_eq!(x.and(y), Ok("different result type"));
    /// # }
    /// ```
    #[inline]
    fn and<U>(self, res: Result<U, E>) -> Result<U, E> {
        async_handle(async {
            match self.await {
                Ok(_) => res,
                Err(e) => Err(e),
            }
        })
    }

    /// Calls `op` if the result is [`Ok`], otherwise returns the [`Err`] value of `self`.
    ///
    ///
    /// This function can be used for control flow based on `Result` values.
    ///
    /// # Examples
    ///
    /// Basic usage:
    ///
    /// ```
    /// use railsgun::BlockInPlaceResult;
    ///
    /// # async fn run() -> () {
    /// fn sq(x: u32) -> Result<u32, u32> { Ok(x * x) }
    /// fn err(x: u32) -> Result<u32, u32> { Err(x) }
    ///
    /// assert_eq!(async {Ok(2u32) as Result<u32,u32>}.and_then(sq).and_then(sq), Ok(16));
    /// assert_eq!(async {Ok(2u32) as Result<u32,u32>}.and_then(sq).and_then(err), Err(4));
    /// assert_eq!(async {Ok(2u32) as Result<u32,u32>}.and_then(err).and_then(sq), Err(2));
    /// assert_eq!(async {Err(3u32) as Result<u32,u32>}.and_then(sq).and_then(sq), Err(3));
    /// # }
    /// ```
    #[inline]
    fn and_then<U, F: FnOnce(T) -> Result<U, E>>(self, op: F) -> Result<U, E> {
        async_handle(async {
            match self.await {
                Ok(t) => op(t),
                Err(e) => Err(e),
            }
        })
    }

    /// Returns `res` if the result is [`Err`], otherwise returns the [`Ok`] value of `self`.
    ///
    /// Arguments passed to `or` are eagerly evaluated; if you are passing the
    /// result of a function call, it is recommended to use [`or_else`], which is
    /// lazily evaluated.
    ///
    /// [`or_else`]: Result::or_else
    ///
    /// # Examples
    ///
    /// Basic usage:
    ///
    /// ```
    /// use railsgun::BlockInPlaceResult;
    ///
    /// # async fn run() {
    /// let x = async {Ok(2u32) as Result<u32,&str>};
    /// let y: Result<u32, &str> = Err("late error");
    /// assert_eq!(x.or(y), Ok(2));
    ///
    /// let x = async {Err("early error") as Result<u32,&str>};
    /// let y: Result<u32, &str> = Ok(2);
    /// assert_eq!(x.or(y), Ok(2));
    ///
    /// let x = async {Err("not a 2") as Result<u32,&str>};
    /// let y: Result<u32, &str> = Err("late error");
    /// assert_eq!(x.or(y), Err("late error"));
    ///
    /// let x = async {Ok(2u32) as Result<u32,&str>};
    /// let y: Result<u32, &str> = Ok(100);
    /// assert_eq!(x.or(y), Ok(2));
    /// # }
    /// ```
    #[inline]
    fn or<F>(self, res: Result<T, F>) -> Result<T, F> {
        async_handle(async {
            match self.await {
                Ok(v) => Ok(v),
                Err(_) => res,
            }
        })
    }

    /// Calls `op` if the result is [`Err`], otherwise returns the [`Ok`] value of `self`.
    ///
    /// This function can be used for control flow based on result values.
    ///
    ///
    /// # Examples
    ///
    /// Basic usage:
    ///
    /// ```
    /// use railsgun::BlockInPlaceResult;
    ///
    /// # async fn run() {
    /// fn sq(x: u32) -> Result<u32, u32> { Ok(x * x) }
    /// fn err(x: u32) -> Result<u32, u32> { Err(x) }
    ///
    /// assert_eq!(async {Ok(2u32) as Result<u32,u32>}.or_else(sq).or_else(sq), Ok(2));
    /// assert_eq!(async {Ok(2u32) as Result<u32,u32>}.or_else(err).or_else(sq), Ok(2));
    /// assert_eq!(async {Err(3u32) as Result<u32,u32>}.or_else(sq).or_else(err), Ok(9));
    /// assert_eq!(async {Err(3u32) as Result<u32,u32>}.or_else(err).or_else(err), Err(3));
    /// # }
    /// ```
    #[inline]
    fn or_else<F, O: FnOnce(E) -> Result<T, F>>(self, op: O) -> Result<T, F> {
        async_handle(async {
            match self.await {
                Ok(t) => Ok(t),
                Err(e) => op(e),
            }
        })
    }

    /// Returns the contained [`Ok`] value or a provided default.
    ///
    /// Arguments passed to `unwrap_or` are eagerly evaluated; if you are passing
    /// the result of a function call, it is recommended to use [`unwrap_or_else`],
    /// which is lazily evaluated.
    ///
    /// [`unwrap_or_else`]: Result::unwrap_or_else
    ///
    /// # Examples
    ///
    /// Basic usage:
    ///
    /// ```
    /// use railsgun::BlockInPlaceResult;
    ///
    /// # async fn run() {
    /// let default = 2;
    /// let x = async {Ok(9u32) as Result<u32, &str>};
    /// assert_eq!(x.unwrap_or(default), 9);
    ///
    ///
    /// let x = async {Err("error") as Result<u32, &str>};
    /// assert_eq!(x.unwrap_or(default), default);
    /// # }
    /// ```
    #[inline]
    fn unwrap_or(self, default: T) -> T {
        async_handle(async {
            match self.await {
                Ok(t) => t,
                Err(_) => default,
            }
        })
    }

    /// Returns the contained [`Ok`] value or computes it from a closure.
    ///
    ///
    /// # Examples
    ///
    /// Basic usage:
    ///
    /// ```
    /// use railsgun::BlockInPlaceResult;
    ///
    /// # async fn run() {
    /// fn count(x: &str) -> u32 { x.len() as u32 }
    ///
    /// assert_eq!(async {Ok(2u32) as Result<u32, &str>}.unwrap_or_else(count), 2);
    /// assert_eq!(async {Err("foo") as Result<u32, &str>}.unwrap_or_else(count), 3);
    /// # }
    /// ```
    #[inline]
    fn unwrap_or_else<F: FnOnce(E) -> T>(self, op: F) -> T {
        async_handle(async {
            match self.await {
                Ok(t) => t,
                Err(e) => op(e),
            }
        })
    }

    /// Returns the contained [`Ok`] value, consuming the `self` value,
    /// without checking that the value is not an [`Err`].
    ///
    /// # Safety
    ///
    /// Calling this method on an [`Err`] is *[undefined behavior]*.
    ///
    /// [undefined behavior]: https://doc.rust-lang.org/reference/behavior-considered-undefined.html
    ///
    /// # Examples
    ///
    /// ```
    /// use railsgun::BlockInPlaceResult;
    ///
    ///
    /// # async fn run() {
    /// assert_eq!(unsafe { async {Ok(2u32) as Result<u32, &str>}.unwrap_unchecked() }, 2);
    /// # }
    /// ```
    ///
    /// ```no_run
    /// use railsgun::BlockInPlaceResult;
    ///
    /// # async fn run() {
    /// unsafe { async {Err("emergency failure") as Result<u32, &str>}.unwrap_unchecked(); } // Undefined behavior!
    /// # }
    /// ```
    #[inline]
    #[track_caller]
    unsafe fn unwrap_unchecked(self) -> T {
        async_handle(async {
            let sync_self = self.await;
            debug_assert!(sync_self.is_ok());
            match sync_self {
                Ok(t) => t,
                // SAFETY: the safety contract must be upheld by the caller.
                Err(_) => unsafe { hint::unreachable_unchecked() },
            }
        })
    }

    /// Returns the contained [`Err`] value, consuming the `self` value,
    /// without checking that the value is not an [`Ok`].
    ///
    /// # Safety
    ///
    /// Calling this method on an [`Ok`] is *[undefined behavior]*.
    ///
    /// [undefined behavior]: https://doc.rust-lang.org/reference/behavior-considered-undefined.html
    ///
    /// # Examples
    ///
    /// ```no_run
    /// use railsgun::BlockInPlaceResult;
    ///
    /// # async fn run() {
    /// unsafe { async {Ok(2) as Result<_, &str>}.unwrap_err_unchecked() }; // Undefined behavior!
    /// # }
    /// ```
    ///
    /// ```
    /// use railsgun::BlockInPlaceResult;
    ///
    /// # async fn run() {
    /// assert_eq!(unsafe { async {Err("emergency failure") as Result<u32, _>}.unwrap_err_unchecked() }, "emergency failure");
    /// # }
    /// ```
    #[inline]
    #[track_caller]
    unsafe fn unwrap_err_unchecked(self) -> E {
        async_handle(async {
            let sync_self = self.await;
            debug_assert!(sync_self.is_err());
            match sync_self {
                // SAFETY: the safety contract must be upheld by the caller.
                Ok(_) => unsafe { hint::unreachable_unchecked() },
                Err(e) => e,
            }
        })
    }
}

fn async_handle<U, F: Future<Output = U>>(future: F) -> U {
    use tokio::runtime::Handle;
    use tokio::task;
    task::block_in_place(move || Handle::current().block_on(future))
}

#[cfg(test)]
mod test {
    use crate::BlockInPlaceResult;

    #[tokio::test(flavor = "multi_thread")]
    async fn test_ok() {
        let x = async { Ok(2u32) as Result<u32, &str> };
        assert_eq!(x.ok(), Some(2));

        let x = async { Err("Nothing here") as Result<u32, _> };
        assert_eq!(x.ok(), None);
    }

    #[tokio::test(flavor = "multi_thread")]
    async fn test_err() {
        let x = async { Ok(2u32) as Result<u32, &str> };
        assert_eq!(x.err(), None);

        let x = async { Err("Nothing here") as Result<u32, &str> };
        assert_eq!(x.err(), Some("Nothing here"));
    }

    #[tokio::test(flavor = "multi_thread")]
    async fn test_map() -> () {
        async fn async_function() -> Result<String, String> {
            Ok("foo".to_string())
        }
        async_function()
            .map(|t| format!("Magic: {}", t))
            .map(|t| println!("{}", t))
            .ok();
    }

    #[tokio::test(flavor = "multi_thread")]
    async fn map_or() -> () {
        let x = async { Ok("foo") as Result<&str, &str> };
        assert_eq!(x.map_or(42, |v| v.len()), 3);

        let x = async { Err("bar") as Result<&str, &str> };
        assert_eq!(x.map_or(42, |v| v.len()), 42);
    }

    #[tokio::test(flavor = "multi_thread")]
    async fn map_or_else() -> () {
        let k = 21;

        let x = async { Ok("foo") as Result<&str, &str> };
        assert_eq!(x.map_or_else(|e| k * 2, |v| v.len()), 3);

        let x = async { Err("bar") as Result<&str, &str> };
        assert_eq!(x.map_or_else(|e| k * 2, |v| v.len()), 42);
    }

    #[tokio::test(flavor = "multi_thread")]
    async fn test_map_err() -> () {
        fn stringify(x: u32) -> String {
            format!("error code: {}", x)
        }

        let x = async { Ok(2u32) as Result<u32, u32> };
        assert_eq!(x.map_err(stringify), Ok(2));

        let x = async { Err(13u32) as Result<u32, u32> };
        assert_eq!(x.map_err(stringify), Err("error code: 13".to_string()));
    }

    #[tokio::test(flavor = "multi_thread")]
    async fn test_and() -> () {
        let x = async { Ok(2u32) as Result<u32, &str> };
        let y: Result<&str, &str> = Err("late error");
        assert_eq!(x.and(y), Err("late error"));

        let x = async { Err("early error") as Result<u32, &str> };
        let y: Result<&str, &str> = Ok("foo");
        assert_eq!(x.and(y), Err("early error"));

        let x = async { Err("not a 2") as Result<u32, &str> };
        let y: Result<&str, &str> = Err("late error");
        assert_eq!(x.and(y), Err("not a 2"));

        let x = async { Ok(2u32) as Result<u32, &str> };
        let y: Result<&str, &str> = Ok("different result type");
        assert_eq!(x.and(y), Ok("different result type"));
    }

    #[tokio::test(flavor = "multi_thread")]
    async fn test_and_then() -> () {
        fn sq(x: u32) -> Result<u32, u32> {
            Ok(x * x)
        }
        fn err(x: u32) -> Result<u32, u32> {
            Err(x)
        }

        assert_eq!(
            async { Ok(2u32) as Result<u32, u32> }
                .and_then(sq)
                .and_then(sq),
            Ok(16)
        );
        assert_eq!(
            async { Ok(2u32) as Result<u32, u32> }
                .and_then(sq)
                .and_then(err),
            Err(4)
        );
        assert_eq!(
            async { Ok(2u32) as Result<u32, u32> }
                .and_then(err)
                .and_then(sq),
            Err(2)
        );
        assert_eq!(
            async { Err(3u32) as Result<u32, u32> }
                .and_then(sq)
                .and_then(sq),
            Err(3)
        );
    }

    #[tokio::test(flavor = "multi_thread")]
    async fn test_or() {
        let x = async { Ok(2u32) as Result<u32, &str> };
        let y: Result<u32, &str> = Err("late error");
        assert_eq!(x.or(y), Ok(2));

        let x = async { Err("early error") as Result<u32, &str> };
        let y: Result<u32, &str> = Ok(2);
        assert_eq!(x.or(y), Ok(2));

        let x = async { Err("not a 2") as Result<u32, &str> };
        let y: Result<u32, &str> = Err("late error");
        assert_eq!(x.or(y), Err("late error"));

        let x = async { Ok(2u32) as Result<u32, &str> };
        let y: Result<u32, &str> = Ok(100);
        assert_eq!(x.or(y), Ok(2));
    }

    #[tokio::test(flavor = "multi_thread")]
    async fn test_or_else() {
        fn sq(x: u32) -> Result<u32, u32> {
            Ok(x * x)
        }
        fn err(x: u32) -> Result<u32, u32> {
            Err(x)
        }

        assert_eq!(
            async { Ok(2u32) as Result<u32, u32> }
                .or_else(sq)
                .or_else(sq),
            Ok(2)
        );
        assert_eq!(
            async { Ok(2u32) as Result<u32, u32> }
                .or_else(err)
                .or_else(sq),
            Ok(2)
        );
        assert_eq!(
            async { Err(3u32) as Result<u32, u32> }
                .or_else(sq)
                .or_else(err),
            Ok(9)
        );
        assert_eq!(
            async { Err(3u32) as Result<u32, u32> }
                .or_else(err)
                .or_else(err),
            Err(3)
        );
    }

    #[tokio::test(flavor = "multi_thread")]
    async fn test_unwrap_or() {
        let default = 2;
        let x = async { Ok(9u32) as Result<u32, &str> };
        assert_eq!(x.unwrap_or(default), 9);

        let x = async { Err("error") as Result<u32, &str> };
        assert_eq!(x.unwrap_or(default), default);
    }

    #[tokio::test(flavor = "multi_thread")]
    async fn test_unwrap_or_else() {
        fn count(x: &str) -> usize {
            x.len()
        }

        assert_eq!(Ok(2).unwrap_or_else(count), 2);
        assert_eq!(Err("foo").unwrap_or_else(count), 3);
    }

    #[tokio::test(flavor = "multi_thread")]
    async fn test_unwrap_unchecked() {
        assert_eq!(
            unsafe { async { Ok(2u32) as Result<u32, &str> }.unwrap_unchecked() },
            2
        )
    }

    #[tokio::test(flavor = "multi_thread")]
    async fn test_unwrap_err_unchecked() {
        assert_eq!(
            unsafe { async { Err("emergency failure") as Result<u32, _> }.unwrap_err_unchecked() },
            "emergency failure"
        );
    }
}
