/*
 * @LastEditors: Ihoey
 * @LastEditTime: 2022-03-20 11:19:03
 * @Email: mail@ihoey.com
 * @Date: 2022-03-19 19:21:22
 * @Author: Ihoey
 */

use std::env;
use std::error::Error;
use std::fs;

pub fn run(config: Config) -> Result<(), Box<dyn Error>> {
  let contents = fs::read_to_string(config.filename)?;

  let result = if config.case_sensitive {
    search(&config.query, &contents)
  } else {
    search_case_insensitive(&config.query, &contents)
  };

  for line in result {
    println!("{}", line);
  }

  return Ok(());
}

pub struct Config {
  pub query: String,
  pub filename: String,
  pub case_sensitive: bool,
}

impl Config {
  pub fn new(mut args: std::env::Args) -> Result<Config, &'static str> {
    if args.len() < 3 {
      return Err("not enough arguments");
    }
    args.next();

    let query = match args.next() {
      Some(arg) => arg,
      None => return Err("Didn't get a query string"),
    };
    let filename = match args.next() {
      Some(arg) => arg,
      None => return Err("Didn't get a file name"),
    };
    let case_sensitive = env::var("CASE_INSENSITIVE").is_err();

    Ok(Config {
      query,
      filename,
      case_sensitive,
    })
  }
}

pub fn search<'a>(query: &str, contents: &'a str) -> Vec<&'a str> {
  // let mut results = Vec::new();
  // for line in contents.lines() {
  //   if line.contains(query) {
  //     results.push(line);
  //   }
  // }
  // results

  contents
    .lines()
    .filter(|line| line.contains(&query))
    .collect()
}

pub fn search_case_insensitive<'a>(query: &str, contents: &'a str) -> Vec<&'a str> {
  // let query = query.to_lowercase();
  // let mut results = Vec::new();
  // for line in contents.lines() {
  //   if line.to_lowercase().contains(&query) {
  //     results.push(line);
  //   }
  // }
  // results

  contents
    .lines()
    .filter(|line| line.to_lowercase().contains(&query))
    .collect()
}

#[cfg(test)]
mod tests {
  use super::*;

  #[test]
  fn one_result() {
    let query = "ihoey";
    let contents = "my name is xixi\n
ihoey
IHOEY
and i am from china";

    assert_eq!(vec!["my name is xixi"], search(query, contents));
  }

  #[test]
  fn case_insensitive() {
    let query = "IHOEY";
    let contents = "my name is xixi\n
ihoey
IHOEY
and i am from china";

    assert_eq!(
      vec!["ihoey", "IHOEY"],
      search_case_insensitive(query, contents)
    );
  }
}
