//! This crate provides an interface to the [SparkFun Qwiic Button LED] over the I2C protocol. It
//! supports manipulating the LED and responding to button presses. It builds on [embedded-hal] and
//! thus supports any platform which implements that crate's traits, like popular microcontrollers
//! and Raspberry Pi models.
//!
//! An example for the Raspberry Pi:
//!
//! ```rust
//! use linux_embedded_hal as hal;
//! use qwiic_button_led::*;
//! use std::{thread, time};
//!
//! // The rPi model 4 B has /dev/i2c-1 as its only I2C device
//! let i2c = hal::I2cdev::new("/dev/i2c-1").unwrap();
//! // The Qwiic Button LED's default address is 0x6F, but is user-configurable
//! let address = 0x6F;
//! let mut button = ButtonLED::init(i2c, address);
//! loop {
//!     let status = button.button_status().unwrap();
//!     if status.pressed {
//!         // if the button is pressed, turn the LED on
//!         button.set_led_brightness(255).unwrap()
//!     } else {
//!         // otherwise, turn it off
//!         button.set_led_brightness(0).unwrap();
//!     }
//!     // sleep a bit to not hammer the I2C bus
//!     thread::sleep(time::Duration::from_millis(10));
//! }
//!
//! ```
//! This example turns the LED on when the button is depressed and turns the LED off when the
//! button is released.
//!
//! The button LED supports both static brightness settings and dynamic pulsing. When setting the
//! brightness statically, the LED stays on. The pulse cycle and pulse off time values configure
//! LED pulsing. The pulse cycle time configures how long the LED is on for while pulsing, and the
//! off time configures how long the LED is off while pulsing.
//!
//! ```rust
//! let mut button = ButtonLED::init(i2c, address);
//! // 300 ms on, 300 ms off, in a loop
//! button.set_led_pulse_cycle_time(300);
//! button.set_led_pulse_off_time(300);
//! ```
//!
//! [SparkFun Qwiic Button LED]: https://www.sparkfun.com/products/15932
//! [embedded-hal]: https://docs.rs/embedded-hal

#![no_std]
#![forbid(unsafe_code)]
#![forbid(missing_docs)]

// based on https://github.com/ferrous-systems/adafruit-seesaw. thanks,
// jamesmunns!

use embedded_hal::blocking::i2c::{Write, WriteRead};

#[repr(u8)]
#[derive(Clone, Copy, Debug)]
enum Register {
    DeviceId = 0x00,
    FirmwareMinor = 0x01,
    ButtonStatus = 0x03,
    InterruptConfig = 0x04,
    ButtonDebounceTime = 0x05,
    PressedQueueStatus = 0x07,
    PressedQueueFront = 0x08,
    PressedQueueBack = 0x0C,
    ClickedQueueStatus = 0x10,
    ClickedQueueFront = 0x11,
    ClickedQueueBack = 0x15,
    LedBrightness = 0x19,
    LedPulseGranularity = 0x1A,
    LedPulseCycleTime = 0x1B,
    LedPulseOffTime = 0x1D,
    I2CAddress = 0x1F,
}

/// The Error crate represents errors the API can encounter.
#[derive(Debug)]
pub enum Error {
    /// An error occurred in I2C communication, e.g. device no longer responding or it sent an
    /// unexpected response.
    I2C,
    /// A runtime error representing incorrect API usage.
    QwiicButton(QwiicButtonError),
}

/// Runtime API errors.
#[derive(Debug)]
pub enum QwiicButtonError {
    /// The address specified for the `set_device_address` function is outside of the valid range
    /// of 0x08 - 0x77.
    AddressError,
}

/// This struct contains all the functions to interface with the Qwiic button LED.
#[derive(Debug)]
pub struct ButtonLED<I2C> {
    i2c: I2C,
    address: u8,
}

/// The button has several status fields which change during operation.
#[derive(Debug)]
pub struct ButtonStatus {
    /// An event (clicked or pressed) is available on the button.
    pub event_available: bool,
    /// The button has been clicked (pressed and released).
    pub clicked: bool,
    /// The button is pressed presently.
    pub pressed: bool,
}

impl From<u8> for ButtonStatus {
    fn from(val: u8) -> Self {
        ButtonStatus {
            event_available: 0b1 & val == 1,
            clicked: 0b10 & val == 2,
            pressed: 0b100 & val == 4,
        }
    }
}

impl From<ButtonStatus> for u8 {
    fn from(val: ButtonStatus) -> u8 {
        (val.event_available as u8 & 0b1)
            | (val.clicked as u8 & 0b1) << 1
            | (val.pressed as u8 & 0b1) << 2
    }
}

/// The button can raise interrupts when pressed and clicked.
#[derive(Debug)]
pub struct InterruptConfig {
    /// The button raises an interrupt when pressed.
    pub pressed: bool,
    /// The button raises an interrupt when clicked (pressed and released).
    pub clicked: bool,
}

impl From<u8> for InterruptConfig {
    fn from(val: u8) -> Self {
        InterruptConfig {
            pressed: 0b01 & val == 1,
            clicked: 0b10 & val == 2,
        }
    }
}

impl From<InterruptConfig> for u8 {
    fn from(val: InterruptConfig) -> u8 {
        (val.clicked as u8 & 0b1) << 1 | (val.pressed as u8 & 0b1)
    }
}

/// The press queue is a 15 element buffer recording timestamps of button presses.
#[derive(Debug)]
pub struct PressedQueueStatus {
    /// Press queue empty.
    pub buffer_empty: bool,
    /// Press queue full.
    pub buffer_full: bool,
}

impl From<u8> for PressedQueueStatus {
    fn from(val: u8) -> Self {
        PressedQueueStatus {
            buffer_empty: 0b010 & val == 2,
            buffer_full: 0b100 & val == 4,
        }
    }
}

/// The click queue is a 15 element buffer recording timestamps of button clicks.
#[derive(Debug)]
pub struct ClickedQueueStatus {
    /// Click queue empty.
    pub buffer_empty: bool,
    /// Click queue full.
    pub buffer_full: bool,
}

impl From<u8> for ClickedQueueStatus {
    fn from(val: u8) -> Self {
        ClickedQueueStatus {
            buffer_empty: 0b010 & val == 2,
            buffer_full: 0b100 & val == 4,
        }
    }
}

const ADDRESS_MIN: u8 = 0x08;
const ADDRESS_MAX: u8 = 0x77;

impl<I2C> ButtonLED<I2C>
where
    I2C: Write + WriteRead,
{
    fn write_u8(&mut self, op: Register, payload: u8) -> Result<(), Error> {
        self.i2c
            .write(self.address, &[op as u8, payload])
            .map_err(|_| Error::I2C)
    }

    fn read_u8(&mut self, op: Register) -> Result<u8, Error> {
        let mut buf: [u8; 1] = [0u8; 1];
        match self.i2c.write_read(self.address, &[op as u8], &mut buf) {
            Ok(_) => Ok(buf[0]),
            Err(_) => Err(Error::I2C),
        }
    }

    fn write_u16(&mut self, op: Register, payload: u16) -> Result<(), Error> {
        let mut cmd: [u8; 3] = [0u8; 3];
        cmd[0] = op as u8;
        cmd[1..].copy_from_slice(&payload.to_le_bytes());
        self.i2c.write(self.address, &cmd).map_err(|_| Error::I2C)
    }

    fn read_u16(&mut self, op: Register) -> Result<u16, Error> {
        let mut buf: [u8; 2] = [0u8; 2];
        match self.i2c.write_read(self.address, &[op as u8], &mut buf) {
            Ok(_) => Ok(u16::from_le_bytes(buf)),
            Err(_) => Err(Error::I2C),
        }
    }

    fn read_u32(&mut self, op: Register) -> Result<u32, Error> {
        let mut buf: [u8; 4] = [0u8; 4];
        match self.i2c.write_read(self.address, &[op as u8], &mut buf) {
            Ok(_) => Ok(u32::from_le_bytes(buf)),
            Err(_) => Err(Error::I2C),
        }
    }

    /// Initialize a ButtonLED struct
    pub fn init(i2c: I2C, address: u8) -> Self {
        Self { i2c, address }
    }

    /// The button ID is read-only and always returns 0x5D.
    pub fn button_id(&mut self) -> Result<u8, Error> {
        self.read_u8(Register::DeviceId)
    }

    /// The firmware version for this button LED.
    pub fn firmware_version(&mut self) -> Result<u16, Error> {
        self.read_u16(Register::FirmwareMinor)
    }

    /// Determine the status of this button.
    pub fn button_status(&mut self) -> Result<ButtonStatus, Error> {
        self.read_u8(Register::ButtonStatus).map(|r| r.into())
    }

    /// Set the button status, e.g. to clear events
    pub fn set_button_status(&mut self, new_status: ButtonStatus) -> Result<(), Error> {
        self.write_u8(Register::ButtonStatus, new_status.into())
    }

    /// Query the interrupt configuration.
    pub fn interrupt_config(&mut self) -> Result<InterruptConfig, Error> {
        self.read_u8(Register::InterruptConfig).map(|r| r.into())
    }

    /// Manipulate the interrupt configuration.
    pub fn set_interrupt_config(&mut self, config: InterruptConfig) -> Result<(), Error> {
        self.write_u8(Register::InterruptConfig, config.into())
    }

    /// Query the debounce time, a measurement of how long after the button is released that it
    /// requires to reset to the default state.
    pub fn button_debounce_time(&mut self) -> Result<u16, Error> {
        self.read_u16(Register::ButtonDebounceTime)
    }

    /// Manipulate the debounce time.
    pub fn set_button_debounce_time(&mut self, time: u16) -> Result<(), Error> {
        self.write_u16(Register::ButtonDebounceTime, time)
    }

    /// Is the button pressed queue full.
    pub fn pressed_queue_full(&mut self) -> Result<bool, Error> {
        self.read_u8(Register::PressedQueueStatus)
            .map(|r| r & 0b100 == 4)
    }

    /// Is the button pressed queue empty.
    pub fn pressed_queue_empty(&mut self) -> Result<bool, Error> {
        self.read_u8(Register::PressedQueueStatus)
            .map(|r| r & 0b010 == 2)
    }

    /// Query the button pressed queue: empty / full.
    pub fn pressed_queue_status(&mut self) -> Result<PressedQueueStatus, Error> {
        self.read_u8(Register::PressedQueueStatus).map(|r| r.into())
    }

    /// The timestamp in milliseconds since the most recent button press in the queue.
    pub fn newest_press_timestamp(&mut self) -> Result<u32, Error> {
        let stamp = self.read_u32(Register::PressedQueueFront)?;
        let mut current_status = self.read_u8(Register::PressedQueueStatus)?;
        current_status |= 1;
        self.write_u8(Register::PressedQueueStatus, current_status)?;
        Ok(stamp)
    }

    /// The timestamp in milliseconds since the first button press in the queue.
    pub fn oldest_press_timestamp(&mut self) -> Result<u32, Error> {
        let stamp = self.read_u32(Register::PressedQueueBack)?;
        let mut current_status = self.read_u8(Register::PressedQueueStatus)?;
        current_status |= 1;
        self.write_u8(Register::PressedQueueStatus, current_status)?;
        Ok(stamp)
    }

    /// Is the button clicked queue full.
    pub fn clicked_queue_full(&mut self) -> Result<bool, Error> {
        self.read_u8(Register::ClickedQueueStatus)
            .map(|r| r & 0b100 == 4)
    }

    /// Is the button clicked queue empty.
    pub fn clicked_queue_empty(&mut self) -> Result<bool, Error> {
        self.read_u8(Register::ClickedQueueStatus)
            .map(|r| r & 0b010 == 2)
    }

    /// Query the button pressed queue: empty / full.
    pub fn clicked_queue_status(&mut self) -> Result<ClickedQueueStatus, Error> {
        self.read_u8(Register::ClickedQueueStatus).map(|r| r.into())
    }

    /// The timestamp in milliseconds since the most recent button click in the queue.
    pub fn newest_click_timestamp(&mut self) -> Result<u32, Error> {
        let stamp = self.read_u32(Register::ClickedQueueFront)?;
        let mut current_status = self.read_u8(Register::ClickedQueueStatus)?;
        current_status |= 1;
        self.write_u8(Register::ClickedQueueStatus, current_status)?;
        Ok(stamp)
    }

    /// The timestamp in milliseconds since the first button click in the queue.
    pub fn oldest_click_timestamp(&mut self) -> Result<u32, Error> {
        let stamp = self.read_u32(Register::ClickedQueueBack)?;
        let mut current_status = self.read_u8(Register::ClickedQueueStatus)?;
        current_status |= 1;
        self.write_u8(Register::ClickedQueueStatus, current_status)?;
        Ok(stamp)
    }

    /// The LED brightness.
    pub fn led_brightness(&mut self) -> Result<u8, Error> {
        self.read_u8(Register::LedBrightness)
    }

    /// Set LED brightness.
    pub fn set_led_brightness(&mut self, brightness: u8) -> Result<(), Error> {
        self.write_u8(Register::LedBrightness, brightness)
    }

    /// The number of steps the LED uses from one brightness setting another when updated.
    pub fn led_pulse_granularity(&mut self) -> Result<u8, Error> {
        self.read_u8(Register::LedPulseGranularity)
    }

    /// Set the LED update granularity.
    pub fn set_led_pulse_granularity(&mut self, brightness: u8) -> Result<(), Error> {
        self.write_u8(Register::LedPulseGranularity, brightness)
    }

    /// When set, the LED pulses on for this value in milliseconds.
    pub fn led_pulse_cycle_time(&mut self) -> Result<u16, Error> {
        self.read_u16(Register::LedPulseCycleTime)
    }

    /// Set the LED pulse on time in milliseconds (0 disables).
    pub fn set_led_pulse_cycle_time(&mut self, cycle_time: u16) -> Result<(), Error> {
        self.write_u16(Register::LedPulseCycleTime, cycle_time)
    }

    /// The duration in milliseconds the LED is off while pulsing.
    pub fn led_pulse_off_time(&mut self) -> Result<u16, Error> {
        self.read_u16(Register::LedPulseOffTime)
    }

    /// Set the LED off time in milliseconds (0 disables).
    pub fn set_led_pulse_off_time(&mut self, off_time: u16) -> Result<(), Error> {
        self.write_u16(Register::LedPulseOffTime, off_time)
    }

    /// The address of this button LED.
    pub fn device_address(&mut self) -> Result<u8, Error> {
        self.read_u8(Register::I2CAddress)
    }

    /// Set the address of this button LED (0x08 - 0x77).
    pub fn set_device_address(&mut self, new_address: u8) -> Result<(), Error> {
        if !(ADDRESS_MIN..=ADDRESS_MAX).contains(&new_address) {
            return Err(Error::QwiicButton(QwiicButtonError::AddressError));
        }
        match self.write_u8(Register::I2CAddress, new_address) {
            Ok(_) => {
                self.address = new_address;
                Ok(())
            }
            Err(e) => Err(e),
        }
    }
}
