This crate provides an interface to the [SparkFun Qwiic Button LED] over the I2C protocol. It
supports manipulating the LED and responding to button presses. It builds on [embedded-hal] and
thus supports any platform which implements that crate's traits, like popular microcontrollers
and Raspberry Pi models.

An example for the Raspberry Pi:

```rust
use linux_embedded_hal as hal;
use qwiic_button_led::*;
use std::{thread, time};

// The rPi model 4 B has /dev/i2c-1 as its only I2C device
let i2c = hal::I2cdev::new("/dev/i2c-1").unwrap();
// The Qwiic Button LED's default address is 0x6F, but is user-configurable
let address = 0x6F;
let mut button = ButtonLED::init(i2c, address);
loop {
    let status = button.button_status().unwrap();
    if status.pressed {
        // if the button is pressed, turn the LED on
        button.set_led_brightness(255).unwrap()
    } else {
        // otherwise, turn it off
        button.set_led_brightness(0).unwrap();
    }
    // sleep a bit to not hammer the I2C bus
    thread::sleep(time::Duration::from_millis(10));
}

```
This example turns the LED on when the button is depressed and turns the LED off when the
button is released.

The button LED supports both static brightness settings and dynamic pulsing. When setting the
brightness statically, the LED stays on. The pulse cycle and pulse off time values configure
LED pulsing. The pulse cycle time configures how long the LED is on for while pulsing, and the
off time configures how long the LED is off while pulsing.

```rust
let mut button = ButtonLED(i2c, address);
// 300 ms on, 300 ms off, in a loop
button.set_led_pulse_cycle_time(300);
button.set_led_pulse_off_time(300);
```

[SparkFun Qwiic Button LED]: https://www.sparkfun.com/products/15932
[embedded-hal]: https://docs.rs/embedded-hal

