use crate::ffi;

#[cfg(feature = "alloc")]
use alloc::vec::Vec;

/// Save the given block of data, overwriting the existing data.
pub fn save(data: &[u8], private: bool) {
    unsafe {
        ffi::qwac_data_save(data.as_ptr(), data.len() as _, if private { 1 } else { 0 });
    }
}

pub fn size() -> usize {
    unsafe { ffi::qwac_data_size() as usize }
}

pub fn shared_size(cartridge_id: u128) -> usize {
    unsafe {
        let cartridge_id = cartridge_id.to_be_bytes();
        ffi::qwac_data_shared_size(cartridge_id.as_ptr()) as usize
    }
}

/// Load the given data into given buffer
pub fn load(data: &mut [u8]) -> usize {
    unsafe { ffi::qwac_data_load(data.as_mut_ptr(), data.len() as _) as usize }
}

pub fn shared_load(cartridge_id: u128, data: &mut [u8]) -> usize {
    unsafe {
        let cartridge_id = cartridge_id.to_be_bytes();
        ffi::qwac_data_shared_load(cartridge_id.as_ptr(), data.as_mut_ptr(), data.len() as _)
            as usize
    }
}

#[cfg(feature = "alloc")]
pub fn load_vec() -> Vec<u8> {
    let mut output = Vec::new();
    let size = size();
    if size > 0 {
        output.resize(size, 0);
        load(&mut output);
    }
    output
}

#[cfg(feature = "alloc")]
pub fn shared_load_vec(cartridge_id: u128) -> Vec<u8> {
    let mut output = Vec::new();
    let size = shared_size(cartridge_id);
    if size > 0 {
        output.resize(size, 0);
        shared_load(cartridge_id, &mut output);
    }
    output
}
