use crate::texture::Texture;
use crate::{ffi, Color, Rectangle};
pub use crate::ffi::CompositeOperation;

#[derive(Debug)]
pub struct Canvas {
    pub(crate) id: u32,
}

impl Canvas {
    pub fn new() -> Self {
        unsafe {
            Self {
                id: ffi::qwac_canvas_create(),
            }
        }
    }

    pub fn resize(&self, width_px: u32, height_px: u32) {
        unsafe {
            ffi::qwac_canvas_resize(self.id, width_px, height_px);
        }
    }

    pub fn clear(&self) {
        unsafe {
            ffi::qwac_canvas_clear(self.id);
        }
    }

    pub fn draw_rectangle(&self, rectangle: Rectangle, color: Color) {
        unsafe {
            ffi::qwac_canvas_draw_rectangle(
                self.id,
                rectangle.x,
                rectangle.y,
                rectangle.width,
                rectangle.height,
                color.r,
                color.g,
                color.b,
                color.a,
            );
        }
    }

    pub fn draw_texture(&self, texture: &Texture, source: Rectangle, destination: Rectangle) {
        unsafe {
            ffi::qwac_canvas_draw_texture(
                self.id,
                texture.id,
                source.x,
                source.y,
                source.width,
                source.height,
                destination.x,
                destination.y,
                destination.width,
                destination.height,
            );
        }
    }

    pub fn draw_canvas(&self, canvas: &Canvas, source: Rectangle, destination: Rectangle) {
        unsafe {
            ffi::qwac_canvas_draw_canvas(
                self.id,
                canvas.id,
                source.x,
                source.y,
                source.width,
                source.height,
                destination.x,
                destination.y,
                destination.width,
                destination.height,
            );
        }
    }

    pub fn as_texture(&self) -> Texture {
        Texture::from_canvas(self)
    }

    pub fn set_global_alpha(&self, alpha: f64) {
        unsafe {
            ffi::qwac_canvas_set_global_alpha(self.id, alpha);
        }
    }

    pub fn set_global_composite_operation(&self, operation: CompositeOperation) {
        unsafe {
            ffi::qwac_canvas_set_global_composite_operation(self.id, operation);
        }
    }
}

impl Drop for Canvas {
    fn drop(&mut self) {
        if self.id != 0 {
            unsafe {
                ffi::qwac_canvas_delete(self.id);
            }
        }
    }
}

pub const SCREEN: Canvas = Canvas { id: 0 };
