//! This is a simple crate that makes making qwac games easier.
//!
//! It has one feature, `alloc`.  You should enable it if you plan on using liballoc (including if
//! `std` is enabled), otherwise, leave it disabled to save some space.

#![no_std]

mod ffi;

pub mod audio;
pub mod event;
pub mod texture;
pub mod data;
pub mod canvas;

#[cfg(feature = "alloc")]
pub extern crate alloc;

#[derive(Default, Clone, Copy, Hash, Debug, Eq, PartialEq, Ord, PartialOrd)]
pub struct Rectangle {
    pub x: i32,
    pub y: i32,
    pub width: i32,
    pub height: i32,
}

#[derive(Default, Clone, Copy, Hash, Debug, Eq, PartialEq, Ord, PartialOrd)]
pub struct Color {
    pub r: u8,
    pub g: u8,
    pub b: u8,
    pub a: u8,
}

pub trait Game {
    /// Create a new Game of this type, starting at the given time in seconds.
    fn new(timestamp: f64) -> Self;

    /// Handle an input event.
    fn event(&mut self, event: event::Event);

    /// Do processing and rendering.  delta tells you the time in seconds between this frame and
    /// the last one.
    fn update(&mut self, frame_milliseconds: u32);
}

#[macro_export]
macro_rules! qwac_game {
    ($game_type:ty) => {
        #[cfg(not(feature = "alloc"))]
        static mut QWAC_RUST_GAME: Option<$game_type> = None;

        #[no_mangle]
        pub extern "C" fn qwac_game_create(timestamp: f64) -> *mut $game_type {
            let game = <$game_type as $crate::Game>::new(timestamp);

            #[cfg(not(feature = "alloc"))]
            {
                unsafe {
                    QWAC_RUST_GAME = None;
                    QWAC_RUST_GAME = Some(game);
                }
                core::ptr::null_mut()
            }

            #[cfg(feature = "alloc")]
            $crate::alloc::boxed::Box::new(game).into_raw()
        }

        #[no_mangle]
        pub extern "C" fn qwac_game_event_gamepad(
            _game: *mut $game_type,
            player: u8,
            button: $crate::event::GamepadButton,
            pressed: u8,
        ) {
            #[cfg(not(feature = "alloc"))]
            let game = unsafe { QWAC_RUST_GAME.as_mut().unwrap() };

            #[cfg(feature = "alloc")]
            let game = unsafe { &mut *_game };

            let state = if pressed == 0 {
                $crate::event::ButtonState::Released
            } else {
                $crate::event::ButtonState::Pressed
            };

            $crate::Game::event(
                game,
                $crate::event::Event::GamepadButton {
                    button,
                    player,
                    state,
                },
            );
        }

        #[no_mangle]
        pub extern "C" fn qwac_game_update(_game: *mut $game_type, frame_milliseconds: u32) {
            #[cfg(not(feature = "alloc"))]
            let game = unsafe { QWAC_RUST_GAME.as_mut().unwrap() };

            #[cfg(feature = "alloc")]
            let game = unsafe { &mut *_game };

            $crate::Game::update(game, frame_milliseconds);
        }

        #[no_mangle]
        pub extern "C" fn qwac_game_destroy(_game: *mut $game_type) {
            #[cfg(not(feature = "alloc"))]
            unsafe {
                QWAC_RUST_GAME = None;
            }

            #[cfg(feature = "alloc")]
            unsafe {
                $crate::alloc::boxed::Box::from_raw(game);
            }
        }
    };
}

pub fn log(string: &str) {
    let pointer = string.as_ptr();
    let len = string.len() as u32;
    unsafe {
        ffi::qwac_log(pointer, len);
    }
}
