use crate::ffi;

pub trait AudioNode {
    fn node_id(&self) -> u32;

    fn connect<A>(&self, target: &A)
    where
        A: AudioNode,
    {
        let id = self.node_id();
        let target = target.node_id();
        unsafe {
            ffi::qwac_audio_connect(id, target);
        }
    }

    fn disconnect<A>(&self, target: &A)
    where
        A: AudioNode,
    {
        let id = self.node_id();
        let target = target.node_id();
        unsafe {
            ffi::qwac_audio_disconnect(id, target);
        }
    }

    fn sink(&self) {
        let id = self.node_id();
        unsafe {
            ffi::qwac_audio_sink(id);
        }
    }

    fn unsink(&self) {
        let id = self.node_id();
        unsafe {
            ffi::qwac_audio_unsink(id);
        }
    }
}

pub trait AudioSource: AudioNode {
    fn play(&self) {
        let id = self.node_id();
        unsafe {
            ffi::qwac_audio_source_play(id);
        }
    }

    fn stop(&self) {
        let id = self.node_id();
        unsafe {
            ffi::qwac_audio_source_stop(id);
        }
    }
}

pub trait SingleValue: AudioNode {
    fn set_value(&mut self, value: f64) {
        let id = self.node_id();
        unsafe {
            ffi::qwac_audio_set_value(id, value);
        }
    }

    fn set_value_at_time(&mut self, time: f64, value: f64) {
        let id = self.node_id();
        unsafe {
            ffi::qwac_audio_set_value_at_time(id, time, value);
        }
    }

    fn linear_ramp_to_value_at_time(&mut self, time: f64, value: f64) {
        let id = self.node_id();
        unsafe {
            ffi::qwac_audio_linear_ramp_to_value_at_time(id, time, value);
        }
    }

    fn exponential_ramp_to_value_at_time(&mut self, time: f64, value: f64) {
        let id = self.node_id();
        unsafe {
            ffi::qwac_audio_exponential_ramp_to_value_at_time(id, time, value);
        }
    }
}

#[derive(Debug, Copy, Clone, PartialEq, Eq, PartialOrd, Ord, Hash)]
pub enum OscillatorType {
    Sine,
    Square,
    Sawtooth,
    Triangle,
}

#[derive(Debug, Clone, PartialEq, Eq, PartialOrd, Ord, Hash)]
pub struct Oscillator {
    slot: u32,
}

impl AudioNode for Oscillator {
    fn node_id(&self) -> u32 {
        self.slot
    }
}

impl SingleValue for Oscillator {}

impl AudioSource for Oscillator {}

impl Oscillator {
    pub fn new() -> Self {
        let slot = unsafe { ffi::qwac_audio_oscillator() };
        Self { slot }
    }

    pub fn set_type(&mut self, type_: OscillatorType) {
        unsafe {
            match type_ {
                OscillatorType::Sine => ffi::qwac_audio_oscillator_sine(self.node_id()),

                OscillatorType::Square => ffi::qwac_audio_oscillator_square(self.node_id()),
                OscillatorType::Sawtooth => ffi::qwac_audio_oscillator_sawtooth(self.node_id()),
                OscillatorType::Triangle => ffi::qwac_audio_oscillator_triangle(self.node_id()),
            }
        }
    }

    pub fn delay(&mut self, seconds: f64) {
        unsafe {
            ffi::qwac_audio_oscillator_delay(self.node_id(), seconds);
        }
    }

    pub fn length(&mut self, seconds: f64) {
        unsafe {
            ffi::qwac_audio_oscillator_length(self.node_id(), seconds);
        }
    }
}

impl Drop for Oscillator {
    fn drop(&mut self) {
        let id = self.node_id();
        unsafe {
            ffi::qwac_audio_delete(id);
        }
    }
}

#[derive(Debug, Clone, PartialEq, Eq, PartialOrd, Ord, Hash)]
pub struct Gain {
    slot: u32,
}

impl AudioNode for Gain {
    fn node_id(&self) -> u32 {
        self.slot
    }
}

impl SingleValue for Gain {}

impl Gain {
    pub fn new() -> Self {
        let slot = unsafe { ffi::qwac_audio_gain() };
        Self { slot }
    }
}

impl Drop for Gain {
    fn drop(&mut self) {
        let id = self.node_id();
        unsafe {
            ffi::qwac_audio_delete(id);
        }
    }
}

#[derive(Debug, Clone, PartialEq, Eq, PartialOrd, Ord, Hash)]
pub struct Delay {
    slot: u32,
}

impl AudioNode for Delay {
    fn node_id(&self) -> u32 {
        self.slot
    }
}

impl SingleValue for Delay {}

impl Delay {
    pub fn new(max_seconds: f64) -> Self {
        let slot = unsafe { ffi::qwac_audio_delay(max_seconds) };
        Self { slot }
    }
}

impl Drop for Delay {
    fn drop(&mut self) {
        let id = self.node_id();
        unsafe {
            ffi::qwac_audio_delete(id);
        }
    }
}
