#![no_std]

pub mod event;
pub mod ffi;

#[derive(Default, Clone, Copy, Hash, Debug, Eq, PartialEq, Ord, PartialOrd)]
pub struct Rectangle {
    pub x: u16,
    pub y: u16,
    pub width: u16,
    pub height: u16,
}

#[derive(Default, Clone, Copy, Hash, Debug, Eq, PartialEq, Ord, PartialOrd)]
pub struct OpaqueColor {
    pub r: u8,
    pub g: u8,
    pub b: u8,
}

#[derive(Default, Clone, Copy, Hash, Debug, Eq, PartialEq, Ord, PartialOrd)]
pub struct Color {
    pub r: u8,
    pub g: u8,
    pub b: u8,
    pub a: u8,
}

pub trait Game {
    /// Create a new Game of this type, starting at the given time in seconds.
    fn new(timestamp: f64) -> Self;

    /// Handle an input event.
    fn event(&mut self, event: event::Event);

    /// Do processing and rendering.  delta tells you the time in seconds between this frame and
    /// the last one.
    fn update(&mut self, delta: f32);
}

#[macro_export]
macro_rules! wag_game {
    ($game_type:ty) => {
        static mut wag_rust_game: Option<$game_type> = None;

        #[no_mangle]
        pub extern "C" fn wag_init(timestamp: f64) {
            unsafe {
                wag_rust_game = None;
                wag_rust_game = Some(<$game_type as $crate::Game>::new(timestamp));
            }
        }

        #[no_mangle]
        pub extern "C" fn wag_event(
            event_type: $crate::ffi::EventType,
            player: u8,
            button: u8,
            x: u16,
            y: u16,
        ) {
            let game = unsafe { wag_rust_game.as_mut().unwrap() };
            match event_type {
                $crate::ffi::EventType::GamepadButtonPress => {
                    if let Some(button) = $crate::event::GamepadButton::from_code(button) {
                        $crate::Game::event(
                            game,
                            $crate::event::Event::GamepadButton {
                                button: button,
                                player: player,
                                state: $crate::event::ButtonState::Pressed,
                            },
                        );
                    }
                }
                $crate::ffi::EventType::GamepadButtonRelease => {
                    if let Some(button) = $crate::event::GamepadButton::from_code(button) {
                        $crate::Game::event(
                            game,
                            $crate::event::Event::GamepadButton {
                                button: button,
                                player: player,
                                state: $crate::event::ButtonState::Released,
                            },
                        );
                    }
                }
                _ => (),
            }
        }

        #[no_mangle]
        pub extern "C" fn wag_update(delta: f32) {
            let game = unsafe { wag_rust_game.as_mut().unwrap() };
            $crate::Game::update(game, delta);
        }

        #[no_mangle]
        pub extern "C" fn wag_shutdown() {
            unsafe {
                wag_rust_game = None;
            }
        }
    };
}

pub fn log(string: &str) {
    let pointer = string.as_ptr();
    let len = string.len() as u32;
    unsafe {
        ffi::wag_log(pointer, len);
    }
}

pub fn display(width_px: u16, height_px: u16) {
    unsafe {
        ffi::wag_display(width_px, height_px);
    }
}

pub fn clear_color(color: OpaqueColor) {
    unsafe {
        ffi::wag_clear_color(color.r, color.g, color.b);
    }
}

pub fn clear() {
    unsafe {
        ffi::wag_clear();
    }
}

pub fn fill_rectangle(rectangle: Rectangle, color: Color) {
    unsafe {
        ffi::wag_fill_rectangle(
            rectangle.x,
            rectangle.y,
            rectangle.width,
            rectangle.height,
            color.r,
            color.g,
            color.b,
            color.a,
        );
    }
}
