use super::*;
use crate::math::{rotate, SQRT_2};

const SQRT_1_2: R = SQRT_2 * 0.5;

pub (crate) struct Op {
    ab_mask: N,
    dagger: bool
}

impl Op {
    #[inline(always)]
    pub fn new(ab_mask: N) -> Self {
        Self{ ab_mask, dagger: false }
    }
}

impl AtomicOp for Op {
    fn atomic_op(&self, psi: &[C], idx: N) -> C {
        if (idx & self.ab_mask).count_ones() & 1 == 1 {
            let psi = (psi[idx], psi[idx ^ self.ab_mask]);
            if self.dagger {
                C { re: SQRT_1_2 * (psi.0.re + psi.1.im),
                    im: SQRT_1_2 * (psi.0.im - psi.1.re) }
            } else {
                C { re: SQRT_1_2 * (psi.0.re - psi.1.im),
                    im: SQRT_1_2 * (psi.0.im + psi.1.re) }
            }
        } else {
            psi[idx]
        }
    }

    fn name(&self) -> String {
        format!("sqrt[iSWAP{}]", self.ab_mask)
    }

    fn is_valid(&self) -> bool {
        self.ab_mask.count_ones() == 2
    }

    fn dgr(self: Ptr<Self>) -> Ptr<dyn AtomicOp> {
        Ptr::new(Self{ dagger: !self.dagger, ..*self })
    }
}

#[cfg(test)] #[test]
fn tests() {
    use crate::operator::single::*;

    const O: C = C{ re: 0.0, im: 0.0 };
    const I: C = C{ re: 1.0, im: 0.0 };
    const i: C = C{ re: 0.0, im: 1.0 };

    let op = SingleOp::from_atomic(Op::new(0b11)).unwrap();
    assert_eq!(op.name(), "sqrt[iSWAP3]");
    assert_eq!(op.matrix(2),
               [   [I,  O,              O,              O],
                   [O,  SQRT_1_2 * I,   SQRT_1_2 * i,   O],
                   [O,  SQRT_1_2 * i,   SQRT_1_2 * I,   O],
                   [O,  O,              O,              I]  ]);
}