use quotemeta::*;
use std::{
    borrow::Cow,
    ffi::{OsStr, OsString},
    os::unix::ffi::OsStrExt,
    path::{Path, PathBuf},
};

fn roundtrip(unquoted: impl AsRef<[u8]>, quoted: impl AsRef<str>) {
    let unquoted = unquoted.as_ref();
    let unquoted = OsStr::from_bytes(unquoted);
    let quoted = quoted.as_ref();

    let test_quoted = quotemeta(&unquoted);
    assert_eq!(quoted, &*test_quoted, "quotemeta(left) != right");
    let test_unquoted = unquotemeta(quoted);
    assert_eq!(unquoted, &*test_unquoted, "unquotemeta(left) != right");
}

#[test]
fn quotemeta_unquotemeta() {
    roundtrip("", "");
    roundtrip("test", "test");
    roundtrip("Hello, world!", "'Hello, world!'");
    roundtrip("isn't", r"$'isn\'t'");
    roundtrip(r"isn\t", r"$'isn\\t'");
    // Octal escapes are unambiguous.
    roundtrip("\n3", r"$'\0123'");
    // Valid UTF-8
    roundtrip("\u{a3}", r"$'\302\243'");
    // Invalid UTF-8 (in this case, Latin-1.)
    roundtrip(b"\xa3", r"$'\243'");
}

// merely a compilation test to ensure that we accept the given types.
#[test]
fn test_types() {
    macro_rules! quotemeta_types {
            ( $( $EXPR:expr ),+ $(,)? ) => {
                $(
                    let _ignore = quotemeta(&$EXPR);
                )+
            }
        }
    macro_rules! unquotemeta_types {
            ( $( $EXPR:expr ),+ $(,)? ) => {
                $(
                    let _ignore = unquotemeta(&$EXPR);
                )+
            }
        }

    quotemeta_types! {
        "",
        String::default(),
        Path::new(""),
        PathBuf::default(),
        OsStr::new(""),
        OsString::default(),
    };
}
