/// Modelled after std::option::Option
/// See: https://doc.rust-lang.org/std/option/enum.Option.html#variant.None
#[derive(Clone,Copy,PartialEq,Debug)]
pub enum Quizas<T> {
    Nada,
    Algo(T),
}

use Quizas::{Algo, Nada};

impl<T> From<Option<T>> for Quizas<T> {
    fn from(x: Option<T>) -> Self {
        match x {
            Some(v) => Algo(v),
            None => Nada,
        }
    }
}

impl<T> Into<Option<T>> for Quizas<T> {
    fn into(self) -> Option<T> {
        match self {
            Algo(x) => Some(x),
            Nada => None,
        }
    }
}

impl<T> Default for Quizas<T> {
    fn default() -> Self {
        Nada
    }
}

impl<T> Quizas<T> {
    pub fn unwrap(self) -> T {
        match self {
            Algo(t) => t,
            Nada => panic!("called `Quizas::unwrap()` on a `Nada` value"),
        }
    }
    pub fn expect(self, msg: &str) -> T {
        match self {
            Algo(t) => t,
            Nada => panic!("{}", msg),
        }
    }
    pub fn is_none(&self) -> bool {
        match self {
            Nada => true,
            _ => false,
        }
    }
    pub fn is_some(&self) -> bool {
        !self.is_none()
    }
    pub fn as_ref(&self) -> Quizas<&T> {
        match *self {
            Algo(ref x) => Algo(x),
            Nada => Nada,
        }
    }
    pub fn as_mut(&mut self) -> Quizas<&mut T> {
        match *self {
            Algo(ref mut x) => Algo(x),
            Nada => Nada,
        }
    }
    pub fn and<U>(self, optb: Quizas<U>) -> Quizas<U> {
        if self.is_none() {
            return Nada
        }
        optb
    }
    pub fn or(self, optb: Quizas<T>) -> Quizas<T> {
        if self.is_some() {
            return self
        }
        optb
    }
    pub fn xor(self, optb: Quizas<T>) -> Quizas<T> {
        if self.is_some() && optb.is_none() {
            return self
        }
        if self.is_none() && optb.is_some() {
            return optb
        }
        Nada 
    }
    pub fn zip<U>(self, other: Quizas<U>) -> Quizas<(T, U)> {
        if self.is_some() && other.is_some() {
            return Algo((self.unwrap(), other.unwrap()))
        }
        Nada
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn it_works() {
        let result = 2 + 2;
        assert_eq!(result, 4);
    }

    #[test]
    fn it_just_works(){
        let _ = Algo(Algo(Algo(0)));
    }

    #[test]
    fn nada_works() {
        let _nada: Quizas<u8> = Nada;
    }

    #[test]
    fn nada_turbofish_works() {
        let _nada: Quizas::<u8> = Nada::<u8>;
    }

    #[test]
    fn algo_works() {
        let _algo: Quizas<u8> = Algo(42);
    }

    #[test]
    fn algo_turbofish_works() {
        let _algo: Quizas::<u8> = Algo::<u8>(42);
    }

    #[test]
    fn algo_from_works() {
        let ft = Some(42);
        let algo = Quizas::from(ft);
        assert_eq!(algo, Algo(42));
    }

    #[test]
    fn nada_from_works() {
        let ft = None::<i32>;
        let algo = Quizas::from(ft);
        assert_eq!(algo, Nada::<i32>);
    }

    #[test]
    fn algo_into_works(){
        let _ft: Option<i32> = Algo::<i32>(42).into();
    }

    #[test]
    fn nada_into_works(){
        let _ft: Option<i32> = Nada::<i32>.into();
    }

    #[test]
    fn default_works(){
        assert_eq!(Nada, Quizas::<i32>::default());
    }

    /*
    #[test]
    fn enum_value(){
        let i = Nada as i32;
        assert_eq!(i, 0);
    }
    */
}
