// $ cargo test --features simd --features fma

use quaternion_core::*;

const PI: f64 = std::f64::consts::PI;
const EPSILON: f64 = 1e-15;  // libmを使う場合は1e-12に落とさないと通らない


// 二つの異なる方法でVersorの軸回りの回転角を求める．
#[test]
fn test_get_angle() {
    // 適当なVersorを作る
    let axis = [0.0, 0.0, 2.0];
    let angle = -1.5 * PI;
    let q = from_axis_angle(axis, angle);
    assert!( ( norm(q) - 1.0 ).abs() < EPSILON );

    // 方法1
    // これが一番シンプル．実部のみを使うので計算量が少ない．
    // 計算精度が一つの変数に依存してしまうのは良くない...？
    // 0 <= angle1 <= 2π
    let angle1 = 2.0 * q.0.acos();

    // 方法2
    // 実部の符号を反映することで幾何学的には方法1と同じ結果が得られる．
    // 実部と虚部両方の値を使っているのでなんとなく気持ちが良い．
    // -π < angle2 <= π
    let angle2 = ( 2.0 * norm_vec(q.1).asin() ).copysign(q.0);

    // 方法3
    // 普通にゼロ除算が発生するが，atanなので計算できる．
    // -π < angle3 < π
    let angle3 = 2.0 * (norm_vec(q.1) / q.0).atan();

    println!("axis: {:?}", normalize_vec(q.1));
    println!("angle1: {}PI, angle2: {}PI", angle1/PI, angle2/PI);

    assert!( (angle1 - 1.5*PI).abs() < EPSILON );
    assert!( (angle2 + 0.5*PI).abs() < EPSILON );
    assert!( (angle3 + 0.5*PI).abs() < EPSILON );
}

#[test]
fn test_axis_angle() {
    // to_axis_angle関数はどのような回転角を入れても正常に変換できるが，
    // テストコードの実装上，angleの範囲は(-2π, 2π)とする．
    let axis = normalize_vec([0.0, 1.0, 1.0]);
    let angle = -1.5*PI;
    let q = from_axis_angle(axis, angle);
    println!("q: {:?}", q);

    assert!( (1.0 - norm(q)).abs() < EPSILON );

    let (re_axis, re_angle) = to_axis_angle(q);

    println!("re_axis: {:?}", re_axis);
    println!("re_angle: {}*PI", re_angle / PI);

    // 軸ベクトルのチェック
    if angle.is_sign_positive() {
        for i in 0..3 {
            assert!( (re_axis[i] - axis[i]).abs() < EPSILON );
        }
    } else {
        // 負の回転角の場合には回転軸が反転する
        for i in 0..3 {
            assert!( (re_axis[i] + axis[i]).abs() < EPSILON );
        }
    }

    // 回転角のチェック
    if angle > PI {
        let tmp = angle - 2.0*PI;
        assert!( (re_angle - tmp).abs() < EPSILON );
    } else if angle < -PI {
        let tmp = -2.0*PI - angle;
        assert!( (re_angle - tmp).abs() < EPSILON );
    } else {
        assert!( (re_angle - angle.abs()).abs() < EPSILON );
    }
}

#[test]
fn test_dcm() {
    // 色々な値でテスト
    let v = [2.5, 1.0, -3.0];
    let diff = [0.2, -0.1, 2.5];
    let mut axis = [1.0, -0.2, 0.9];
    for i in 0..20 {
        axis = add_vec(axis, diff);
        let q = from_axis_angle(axis, PI * (i as f64));  // Versor
        let dcm = to_dcm(q);
        let q_rest = from_dcm(dcm);

        assert!( ( norm(q_rest) - 1.0 ).abs() < EPSILON );
    
        let rotated_q = vector_rotation(q, v);
        let rotated_q_rest = vector_rotation(q_rest, v);
    
        assert!( (rotated_q[0] - rotated_q_rest[0]).abs() < EPSILON );
        assert!( (rotated_q[1] - rotated_q_rest[1]).abs() < EPSILON );
        assert!( (rotated_q[2] - rotated_q_rest[2]).abs() < EPSILON );
    }

    // a <--> b の相互変換が正しく行えるかテスト
    let a = [1.0f64, 0.0, 0.0];
    let b = normalize_vec([0.0f64, 1.0, 1.0]);
    let q = rotate_a_to_b(a, b);

    let m_a2b = to_dcm(q);
    let b_check = matrix_product(m_a2b, a);
    assert!( (b[0] - b_check[0]).abs() < EPSILON );
    assert!( (b[1] - b_check[1]).abs() < EPSILON );
    assert!( (b[2] - b_check[2]).abs() < EPSILON );

    let m_b2a = to_dcm( conj(q) );
    let a_check = matrix_product(m_b2a, b);
    assert!( (a[0] - a_check[0]).abs() < EPSILON );
    assert!( (a[1] - a_check[1]).abs() < EPSILON );
    assert!( (a[2] - a_check[2]).abs() < EPSILON );
}

#[test]
fn test_euler() {
    let yaw_ori   = PI / 4.0;
    let pitch_ori = PI / 4.0;
    let roll_ori  = PI / 4.0;
    let q = from_euler_angles([yaw_ori, pitch_ori, roll_ori]);
    
    let [yaw, pitch, roll] = to_euler_angles(q);
    assert!((yaw   - yaw_ori).abs()   < EPSILON);
    assert!((pitch - pitch_ori).abs() < EPSILON);
    assert!((roll  - roll_ori).abs()  < EPSILON);
}

#[test]
fn test_rotation_vector() {
    // 適当なVersorを作る
    let q = from_axis_angle([1.0, 2.0, 3.0], PI);

    // 回転ベクトルに変換
    let rot_v = to_rotation_vector(q);

    // 回転ベクトルから復元したVecsor
    let mut q_rest = from_rotation_vector(rot_v);

    // 符号が反転していても，３次元空間上で表す回転は同じ
    if q.0.is_sign_positive() && q_rest.0.is_sign_negative() {
        q_rest = negate(q_rest);
    }

    let diff = sub(q, q_rest);
    assert!(diff.0.abs() < EPSILON);
    for i in 0..3 {
        assert!(diff.1[i].abs() < EPSILON);
    }
}

#[test]
fn test_cross() {
    let r1 = [1.0, 0.0, 0.0];
    let r2 = [0.0, 1.0, 0.0];

    let r = cross_vec(r1, r2);
    assert_eq!( [0.0, 0.0, 1.0] , r );
}

#[test]
fn test_add() {
    let a: Quaternion<f64> = (0.5, [1.0, 1.0, 1.0]);
    let b: Quaternion<f64> = (0.5, [1.0, 1.0, 1.0]);
    assert_eq!( add(a, b), (1.0, [2.0; 3]) );
}

#[test]
fn test_sub() {
    let a = (0.5, [1.0, 1.0, 1.0]);
    let b = (0.5, [1.0, 1.0, 1.0]);
    assert_eq!( sub(a, b), (0.0, [0.0; 3]) );
}

#[test]
fn test_scale_add() {
    let s = 2.0_f64;
    let a = (0.5, [1.0, 2.5, 1.2]);
    let b = (2.2, [6.5, 1.0, 3.4]);

    let result_1 = scale_add(s, a, b);
    let result_2 = add( scale(s, a), b );
    let diff = sub(result_1, result_2);
    assert!( diff.0.abs() < EPSILON );
    assert!( diff.1[0].abs() < EPSILON );
    assert!( diff.1[1].abs() < EPSILON );
    assert!( diff.1[2].abs() < EPSILON );
}

#[test]
fn test_hadamard() {
    let a = (1.0f64, [2.0, 3.0, 4.0]);
    let b = (4.0f64, [3.0, 2.0, 1.0]);

    let result = hadamard(a, b);
    assert!((result.0 - 4.0).abs() < EPSILON);
    assert!((result.1[0] - 6.0).abs() < EPSILON);
    assert!((result.1[1] - 6.0).abs() < EPSILON);
    assert!((result.1[2] - 4.0).abs() < EPSILON);
}

#[test]
fn test_norm() {
    let q = from_axis_angle([1.0, 2.0, 0.5], 1.5);
    assert!( (norm(q) - 1.0f64).abs() < EPSILON);
}

#[test]
fn test_normalize() {
    let q = (1.0, [2.0, 3.0, 4.0]);
    let q_n = normalize(q);
    assert!( (norm(q_n) - 1.0f64).abs() < EPSILON );
}

#[test]
fn test_negate() {
    let q = (-1.0, [1.0, 2.0, -1.0]);
    let p = negate(q);

    assert_eq!( p, (1.0, [-1.0, -2.0, 1.0]) )
}

#[test]
fn test_mul() {
    let a = (1.0f64, [0.5, -1.2, 3.0]);
    let b = (0.5f64, [-0.2, 2.5, -3.3]);

    let v0 = scale_vec(a.0, b.1);
    let v1 = scale_vec(b.0, a.1);
    let p = (
        a.0 * b.0 - dot_vec(a.1, b.1),
        add_vec( add_vec(v0, v1), cross_vec(a.1, b.1) )
    );

    let q = mul(a, b);
    assert!( (p.0 - q.0).abs() < EPSILON );
    for i in 0..3 {
        assert!( (p.1[i] - q.1[i]).abs() < EPSILON );
    }
}

// 手計算した結果で動作確認
#[test]
fn test_vector_rotation() {
    let r: Vector3<f64> = [2.0, 2.0, 0.0];
    let q = from_axis_angle([0.0, 1.0, 0.0], PI/2.0);
    let result = vector_rotation(q, r);

    let diff = sub_vec(result, [0.0, 2.0, -2.0]);
    for i in 0..3 {
        assert!( diff[i].abs() < EPSILON );
    }
}

#[test]
fn test_frame_rotation() {
    let r = [2.0, 2.0, 0.0];
    let q = from_axis_angle([0.0, 1.0, 0.0], PI/2.0);
    let result = frame_rotation(q, r);

    let diff = sub_vec(result, [0.0, 2.0, 2.0]);
    for i in 0..3 {
        assert!( diff[i].abs() < EPSILON );
    }
}

// 回転軸と回転角を取り出す
#[test]
fn test_to_axis_angle() {
    let axis = [1.0, 4.0, 2.0];
    let angle = PI/4.0;
    let q = from_axis_angle(axis, angle);

    // 軸の方向は分かるが，元の大きさはわからない．
    let n = normalize_vec(axis);
    let f = to_axis_angle(q);
    assert!( (f.1 - angle).abs() < EPSILON );
    for i in 0..3 {
        assert!( (f.0[i] - n[i]).abs() < EPSILON );
    }
}

#[test]
fn test_rotate_a_to_b() {
    let a = normalize_vec([1.0f64, -0.5, -2.0]);
    
    let q = from_axis_angle([1.0, 0.5, -0.5], PI);
    let b = vector_rotation(q, a);

    let a_to_b = rotate_a_to_b(a, b);
    let a_to_b_t = rotate_a_to_b_param(a, b, 1.0);
    println!("a_to_b: {:?}", a_to_b);
    let b_rest = vector_rotation(a_to_b, a);
    let b_rest_t = vector_rotation(a_to_b_t, a);

    let diff = sub_vec(b, b_rest);
    let diff_t = sub_vec(b, b_rest_t);
    for i in 0..3 {
        assert!(diff[i].abs() < EPSILON, "{}", diff[i]);
        assert!(diff_t[i].abs() < EPSILON, "{}", diff_t[i]);
    }
}