use crossterm::{cursor::MoveTo, event, terminal, ExecutableCommand};
use dirs::home_dir;
use serde::{Deserialize, Serialize};
use text_io::read;

use egg_mode::{
    auth::{access_token, authorize_url, request_token},
    error::{Error::TwitterError, TwitterErrors},
    tweet::{like, mentions_timeline, Tweet},
    KeyPair, Token,
};

use std::{
    clone::Clone,
    fs,
    io::{stdout, Stdout, Write},
    iter::Iterator,
    process::exit,
};

use tui::{
    backend::{Backend, CrosstermBackend},
    layout::{Constraint, Layout},
    style,
    text::{Span, Spans},
    widgets::{Block, Borders, Paragraph},
    Terminal,
};

#[derive(Clone, Deserialize, Serialize)]
struct Keys {
    key: String,
    secret: String,
    access_token: Option<String>,
    access_token_secret: Option<String>,
}

type TwitterResult = Result<(), TwitterErrors>;

const MENU_TEXT: &str = "(l)ike | (n)ext | e(x)it";

async fn authenticate() -> Token {
    let key_file = home_dir().unwrap().join(".config/qt/keys.toml");
    let key_map: Keys = toml::from_str(&fs::read_to_string(&key_file).unwrap()).unwrap();
    if key_map.access_token.is_some() {
        let consumer_pair = KeyPair::new(key_map.key, key_map.secret);
        let access_token_pair = KeyPair::new(
            key_map.access_token.unwrap(),
            key_map.access_token_secret.unwrap(),
        );

        return Token::Access {
            consumer: consumer_pair,
            access: access_token_pair,
        };
    }
    let mut config_file = key_map.clone();
    let consumer_pair = KeyPair::new(key_map.key, key_map.secret);

    let auth_token = request_token(&consumer_pair, "oob").await.unwrap();
    let auth_url = authorize_url(&auth_token);
    print!(
        "\nGo to this URL to authenticate: {}\n\nEnter PIN: ",
        auth_url
    );
    stdout().flush().unwrap();
    let verifier: String = read!();
    println!();
    let (token, _, _) = access_token(consumer_pair, &auth_token, verifier)
        .await
        .unwrap();

    match token {
        Token::Access {
            consumer: _,
            access: ref a,
        } => {
            config_file.access_token = Some(a.key.to_string());
            config_file.access_token_secret = Some(a.secret.to_string());
        }
        Token::Bearer(_) => panic!(),
    };
    let key_toml = toml::to_string(&config_file).unwrap();
    fs::write(key_file, key_toml).unwrap();

    token
}

async fn get_mentions(token: &Token) -> Vec<Tweet> {
    mentions_timeline(&token).start().await.unwrap().1.to_vec()
}

async fn send_like(tweet: &Tweet, token: &Token) -> TwitterResult {
    match like(tweet.id, &token).await {
        Ok(_) => Ok(()),
        Err(TwitterError(_, e)) => Err(e),
        Err(_) => panic!(),
    }
}

// async fn send_reply(tweet: &Tweet, reply_text: String, token: &Token) {
//     // TODO should mention everyone in the thread, not just who was directly replied to
//     let reply_text = format!(
//         "@{} {}",
//         tweet.user.as_ref().unwrap().screen_name,
//         reply_text
//     );
//     DraftTweet::new(reply_text)
//         .in_reply_to(tweet.id)
//         .send(&token)
//         .await
//         .unwrap();
// }

fn err_view(errs: TwitterErrors) -> String {
    errs.errors[0].message.clone()
}

fn tweet_view<'a>(tweet: &'a Tweet) -> Vec<Spans<'a>> {
    let screen_name = &tweet.user.as_ref().unwrap().screen_name;
    let screen_name_msg = format!("@{} says:\n", screen_name);
    let in_response_msg = match &tweet.in_reply_to_screen_name {
        None => "".to_string(),
        Some(name) => format!("in response to @{}\n", name),
    };

    vec![
        Spans::from(Span::styled(
            screen_name_msg,
            style::Style::default().add_modifier(style::Modifier::BOLD),
        )),
        Spans::from(Span::styled(
            in_response_msg,
            style::Style::default().add_modifier(style::Modifier::ITALIC),
        )),
        Spans::from(Span::raw(&tweet.text)),
    ]
}

fn get_key() -> char {
    loop {
        match event::read().unwrap() {
            event::Event::Key(event) => match event.code {
                event::KeyCode::Char(c) => return c,
                _ => (),
            },
            _ => (),
        }
    }
}

async fn run_action(tweet: &Tweet, key: char, token: &Token) -> String {
    if key == 'x' {
        stdout()
            .execute(terminal::Clear(terminal::ClearType::All))
            .unwrap()
            .execute(MoveTo(0, 0))
            .unwrap();
        exit(0);
    };
    if key == 'l' {
        match send_like(tweet, token).await {
            Ok(()) => "Success".to_string(),
            Err(e) => err_view(e),
        }
    } else {
        "You can't do that".to_string()
    }
}

fn draw(text_0: Vec<Spans>, text_1: Vec<Spans>, terminal: &mut Terminal<CrosstermBackend<Stdout>>) {
    terminal
        .draw(|f| {
            let layout = Layout::default()
                .constraints([
                    Constraint::Length(3),
                    Constraint::Min(5),
                    Constraint::Length(3),
                ])
                .split(f.size());

            f.render_widget(
                Paragraph::new(vec![Spans::from(MENU_TEXT)])
                    .block(Block::default().borders(Borders::ALL)),
                layout[0],
            );
            f.render_widget(
                Paragraph::new(text_0).block(Block::default().borders(Borders::ALL)),
                layout[1],
            );
            f.render_widget(
                Paragraph::new(text_1).block(Block::default().borders(Borders::ALL)),
                layout[2],
            );
        })
        .unwrap();
    let height = terminal::size().unwrap().1;
    stdout().execute(MoveTo(3, height - 2)).unwrap();
}

#[tokio::main]
async fn main() {
    let mut term_backend = CrosstermBackend::new(stdout());
    term_backend.clear().unwrap();
    let mut terminal = Terminal::new(term_backend).unwrap();

    draw(
        vec![Spans::from("Authenticating...")],
        vec![],
        &mut terminal,
    );
    let token = authenticate().await;
    draw(
        vec![Spans::from("Getting mentions...")],
        vec![],
        &mut terminal,
    );
    let tweets = get_mentions(&token).await;

    terminal::enable_raw_mode().unwrap();
    for tweet in tweets {
        draw(tweet_view(&tweet), vec![Spans::from("")], &mut terminal);
        loop {
            let key = get_key();
            let message = match key {
                'n' => break,
                c => {
                    draw(
                        tweet_view(&tweet),
                        vec![Spans::from("Working...")],
                        &mut terminal,
                    );
                    run_action(&tweet, c, &token).await
                }
            };
            draw(
                tweet_view(&tweet),
                vec![Spans::from(message)],
                &mut terminal,
            );
        }
    }
}
