use ansi_term::Style;
use crossterm::{cursor::MoveLeft, ExecutableCommand};
use dirs::home_dir;
use serde::{Deserialize, Serialize};
use text_io::read;
use tokio::{join, spawn};

use egg_mode::{
    auth::{access_token, authorize_url, request_token},
    error::{Error::TwitterError, TwitterErrors},
    tweet::{like, mentions_timeline, DraftTweet, Tweet},
    KeyPair, Token,
};

use std::{
    clone::Clone,
    fs,
    future::Future,
    io::{stdout, Write},
    process::exit,
};

#[derive(Clone, Deserialize, Serialize)]
struct Keys {
    key: String,
    secret: String,
    access_token: Option<String>,
    access_token_secret: Option<String>,
}

type TwitterResult = Result<(), TwitterErrors>;

async fn authenticate() -> Token {
    let auth_msg = Style::new().italic().paint("Authenticating...");
    println!("{}", auth_msg);
    let key_file = home_dir().unwrap().join(".config/qt/keys.toml");
    let key_map: Keys = toml::from_str(&fs::read_to_string(&key_file).unwrap()).unwrap();
    if key_map.access_token.is_some() {
        let consumer_pair = KeyPair::new(key_map.key, key_map.secret);
        let access_token_pair = KeyPair::new(
            key_map.access_token.unwrap(),
            key_map.access_token_secret.unwrap(),
        );

        return Token::Access {
            consumer: consumer_pair,
            access: access_token_pair,
        };
    }
    let mut config_file = key_map.clone();
    let consumer_pair = KeyPair::new(key_map.key, key_map.secret);

    let auth_token = request_token(&consumer_pair, "oob").await.unwrap();
    let auth_url = authorize_url(&auth_token);
    print!(
        "\nGo to this URL to authenticate: {}\n\nEnter PIN: ",
        auth_url
    );
    stdout().flush().unwrap();
    let verifier: String = read!();
    println!();
    let (token, _, _) = access_token(consumer_pair, &auth_token, verifier)
        .await
        .unwrap();

    match token {
        Token::Access {
            consumer: _,
            access: ref a,
        } => {
            config_file.access_token = Some(a.key.to_string());
            config_file.access_token_secret = Some(a.secret.to_string());
        }
        Token::Bearer(_) => panic!(),
    };
    let key_toml = toml::to_string(&config_file).unwrap();
    fs::write(key_file, key_toml).unwrap();

    token
}

async fn get_mentions(token: &Token) -> Vec<Tweet> {
    println!("{}\n", Style::new().italic().paint("Getting mentions..."));
    mentions_timeline(&token).start().await.unwrap().1.to_vec()
}

async fn send_like(tweet: &Tweet, token: &Token) -> TwitterResult {
    match like(tweet.id, &token).await {
        Ok(_) => Ok(()),
        Err(TwitterError(_, e)) => Err(e),
        Err(_) => panic!(),
    }
}

async fn send_reply(tweet: &Tweet, reply_text: String, token: &Token) {
    // TODO should mention everyone in the thread, not just who was directly replied to
    let reply_text = format!(
        "@{} {}",
        tweet.user.as_ref().unwrap().screen_name,
        reply_text
    );
    DraftTweet::new(reply_text)
        .in_reply_to(tweet.id)
        .send(&token)
        .await
        .unwrap();
}

fn show_err(errs: TwitterErrors) {
    stdout().execute(MoveLeft(2)).unwrap();
    for err in errs.errors {
        println!("{}", Style::new().italic().paint(err.message))
    }
    print!("\n> ");
    stdout().flush().unwrap();
}

fn show_tweet(tweet: &Tweet) {
    let screen_name = &tweet.user.as_ref().unwrap().screen_name;
    let screen_name_msg = Style::new().bold().paint(format!("@{} says:", screen_name));
    let in_response_msg = match &tweet.in_reply_to_screen_name {
        None => Style::new().paint(""),
        Some(name) => {
            let reply_to_msg = format!("in response to @{}\n", name);
            Style::new().italic().paint(reply_to_msg)
        }
    };
    println!("{}\n{}{}\n", screen_name_msg, in_response_msg, tweet.text);
}

fn prompt() -> String {
    print!("> ");
    stdout().flush().unwrap();
    read!("{}\n")
}

async fn process_actions(tweet: &Tweet, actions: String, token: &Token) -> bool {
    let mut to_like = false;
    for action in actions.chars() {
        if action == 'l' {
            to_like = true;
        } else if action == 'r' {
            let reply_prompt = Style::new().italic().paint("Your reply: ");
            print!("{}", reply_prompt);
            stdout().flush().unwrap();

            let reply_text = read!();
            send_reply(&tweet, reply_text, &token).await;
            println!();
        } else if action == 'x' {
            exit(0);
        }
    }

    to_like
}

async fn run_action(action: impl Future<Output = TwitterResult>) {
    match action.await {
        Ok(()) => (),
        Err(e) => show_err(e),
    };
}

#[tokio::main]
async fn main() {
    let token = authenticate().await;
    let tweets = get_mentions(&token).await;
    let mut actions;
    let mut to_like;
    let mut next_tweet_to_like = None;

    for tweet in tweets {
        show_tweet(&tweet);
        let prompt = spawn(async { prompt() });
        if next_tweet_to_like.is_some() {
            let tweet_to_like = next_tweet_to_like.unwrap();
            actions = join!(prompt, run_action(send_like(&tweet_to_like, &token)))
                .0
                .unwrap();
        } else {
            actions = prompt.await.unwrap();
        }

        println!();
        to_like = process_actions(&tweet, actions, &token).await;
        next_tweet_to_like = if to_like { Some(tweet) } else { None };
    }
}
