#!/bin/bash

# This script does some very basic benchmarks with 'qsv' on a city population
# data set (which is a strict subset of the `worldcitiespop` data set). If it
# doesn't exist on your system, it will be downloaded to /tmp for you.
#
# These aren't meant to be overly rigorous, but they should be enough to catch
# significant regressions.
#
# Make sure you're using an `qsv` generated by `cargo build --release`.

set -e

pat="$1"
datazip=worldcitiespop_mil.zip
data=worldcitiespop_mil.csv
countrydata=countrynames.csv
data_idx=worldcitiespop_mil.csv.idx
data_to_exclude=data_to_exclude.csv
searchset_patterns=searchset_patterns.txt
nyc311datazip=nyc-311-sample-100k.zip
nyc311data=nyc-311-sample-100k.csv
if [ ! -r "$data" ]; then
  printf "Downloading benchmarking data...\n"
  curl -sS https://raw.githubusercontent.com/wiki/jqnatividad/qsv/files/worldcitiespop_mil.zip > "$datazip"
  unzip "$datazip"
  qsv sample --seed 42 50000 "$data" -o "$data_to_exclude"
  printf "santa\nfort\ncamp\n" > "$searchset_patterns"
  curl -sS https://raw.githubusercontent.com/wiki/jqnatividad/qsv/files/nyc-311-sample-100k.zip > "$nyc311datazip"
  unzip "$nyc311datazip"
fi
os_type=$(echo $OSTYPE | cut -c 1-6)
if [[ "$os_type" == "darwin" ]]; then
  data_size=$(stat -f '%z' "$data")
  nyc311data_size=$(stat -f '%z' "$nyc311data")
else
  data_size=$(stat --format '%s' "$data")
  nyc311data_size=$(stat --format '%s' "$nyc311data")
fi
if [ ! -r "$countydata" ]; then
  curl -sS https://gist.githubusercontent.com/anonymous/063cb470e56e64e98cf1/raw/98e2589b801f6ca3ff900b01a87fbb7452eb35c7/countrynames.csv > "$countrydata"
fi

function real_seconds {
  cmd=$(echo $@ "> /dev/null 2>&1")
  t=$(
    $(which time) -p sh -c "$cmd" 2>&1 \
      | grep '^real' \
      | awk '{print $2}')
  if [ $(echo "$t < 0.01" | bc) = 1 ]; then
    t=0.01
  fi
  echo $t
}

function benchmark {
  rm -f "$data_idx"
  t1=$(real_seconds "$@")
  rm -f "$data_idx"
  t2=$(real_seconds "$@")
  rm -f "$data_idx"
  t3=$(real_seconds "$@")
  echo "scale=2; ($t1 + $t2 + $t3) / 3" | bc
}

function benchmark_with_index {
  rm -f "$data_idx"
  qsv index "$data"
  t1=$(real_seconds "$@")
  t2=$(real_seconds "$@")
  t3=$(real_seconds "$@")
  rm -f "$data_idx"
  echo "scale=2; ($t1 + $t2 + $t3) / 3" | bc
}

function run {
  index=
  nyc311=
  while true; do
    case "$1" in
      --index) index="yes" && shift ;;
      --nyc311) nyc311="yes" && shift ;;
      *) break ;;
    esac
  done
  name="$1"
  shift

  printf "%-25s" "$name"
  if [ -z "$pat" ] || echo "$name" | grep -E -q "^$pat$"; then
    if [ -z "$index" ]; then
      t=$(benchmark "$@")
    else
      t=$(benchmark_with_index "$@")
    fi
    if [ -z "$nyc311" ]; then
      mb_per=$(echo "scale=2; ($data_size / $t) / 2^20" | bc)
      recs_per=$(echo "scale=2; (1000000 / $t)" | bc)
    else
      mb_per=$(echo "scale=2; ($nyc311data_size / $t) / 2^20" | bc)
      recs_per=$(echo "scale=2; (100000 / $t)" | bc)
    fi
    mb_per=$(printf "%0.02f" $mb_per)
    recs_per=$(printf "%'.2f" $recs_per)
    printf -v tprint "%0.02f" $t
    printf "%-11s%-12s%-12s\n" "$tprint" "$mb_per" "$recs_per"
    printf "%s\t%0.02f\t%s\t%s\n" $name $t $mb_per $recs_per >> $benchmarkfile
  fi
}

qsvver=$(qsv --version)
current_time=$(date "+%Y-%m-%d-%H-%M-%S")
# printf "Scrambling benchmark data...\n"
# qsv index "$data"
# qsv scramble "$data" > temp.csv
# mv -f temp.csv "$data"
benchmarkfile=qsvbench-$qsvver-$current_time.tsv
printf "%-25s%-11s%-12s%-12s\n" BENCHMARK TIME_SECS MB_PER_SEC RECS_PER_SEC
printf "benchmark\ttime_secs\tmb_per_sec\trecs_per_sec\n" > $benchmarkfile
run apply_op_string qsv apply operations trim,upper Country "$data"
run apply_op_similarity qsv apply operations lower,simdln Country --comparand union "$data"
run apply_op_soundex qsv apply operations lower,soundex City --comparand boston "$data" 
run --nyc311 apply_datefmt qsv apply datefmt \"Created Date\" "$nyc311data"
run --nyc311 apply_emptyreplace qsv apply emptyreplace \"Bridge Highway Name\" --replacement Unspecified "$nyc311data"
run --nyc311 apply_geocode qsv apply geocode Location --new-column geocoded_location -q "$nyc311data"
run count qsv count "$data"
run --index count_index qsv count "$data"
run dedup qsv dedup "$data"
run enum qsv enum "$data"
run exclude qsv exclude Country "$data" Country "$data_to_exclude"
run --index exclude_index qsv exclude Country "$data" Country "$data_to_exclude"
run explode qsv explode City "-" "$data"
run fill qsv fill -v Unknown Population "$data"
run fixlengths qsv fixlengths "$data"
run flatten qsv flatten "$data"
run flatten_condensed qsv flatten "$data" --condense 50
run fmt qsv fmt --crlf "$data"
run frequency qsv frequency "$data"
run --index frequency_index qsv frequency "$data"
run frequency_selregex qsv frequency -s /^R/ "$data"
run index qsv index "$data"
run join qsv join --no-case Country "$data" Abbrev "$countrydata"
run lua qsv lua map pop_empty "tonumber\(Population\)==nil" "$data"
run partition qsv partition Region /tmp/partitioned "$data"
run rename qsv rename 'country,city,accent_city,region,population,lat,long' "$data"
run reverse qsv reverse "$data"
run sample_10 qsv sample 10 "$data"
run --index sample_10_index qsv sample 10 "$data"
run sample_1000 qsv sample 1000 "$data"
run --index sample_1000_index qsv sample 1000 "$data"
run sample_100000 qsv sample 100000 "$data"
run --index sample_100000_index qsv sample 100000 "$data"
run --index sample_25pct_index qsv sample 0.25 "$data"
run --index scramble_index qsv scramble "$data"
run search qsv search -s Country "'(?i)us'" "$data"
run searchset qsv searchset "$searchset_patterns" "$data"
run select qsv select 'Country,City' "$data"
run select_regex qsv select /^L/ "$data"
run slice_one_middle qsv slice -i 500000 "$data"
run --index slice_one_middle_index qsv slice -i 500000 "$data"
run sort qsv sort -s Country "$data"
run split qsv split --size 50000 split_tempdir "$data"
run --index split_index qsv split --size 50000 split_tempdir "$data"
run stats qsv stats "$data"
run --index stats_index qsv stats "$data"
run stats_everything qsv stats "$data" --everything
run --index stats_everything_index qsv stats "$data" --everything
run table qsv table "$data"
run transpose qsv transpose "$data"
