//!
//! ```
//! let x = 1;
//! let s = stringify!{
//!     "a" => x;
//!     "b" => 2usize;
//!     "c" => 3f64;
//!     "d" => x + 6;
//!     "e" => vec![4i32, 5i32, 6i32];
//!     "f" => "hello world";
//!     "g" => Some(8);
//!     "h" => y
//! };
//! assert_eq!(s, "a=1&b=2&c=3&d=7&e[]=4&e[]=5&e[]=6&f=hello%20world");
//!
//! use std::collections::BTreeMap;
//! let mut m = BTreeMap::new();
//! m.insert("a", 1);
//! m.insert("b", 2);
//! m.insert("c", 3);
//! let s = stringify!(m);
//! assert_eq!(s,  "a=1&b=2&c=3");
//!
//! let s = stringify![
//!     ("a", 1),
//!     ("b", 2),
//!     ("c", 3)
//! ];
//! assert_eq!(s,  "a=1&b=2&c=3");
//! ```

mod qs;

pub use crate::qs::QueryString;
pub use crate::qs::UrlEncoding;

#[macro_export]
macro_rules! stringify {
    
    ($(($k:expr, $v:expr)),+) => {{
        let mut q_vec: Vec<String> = Vec::new();
        $(
            q_vec.push($v.to_qs($k));
        )*
        q_vec.join("&")
    }};
    ($m:expr) => {{
        let mut q_vec: Vec<String> = Vec::new();
        for (k, v) in $m.into_iter() {
            q_vec.push(v.to_qs(k));
        }
        q_vec.join("&")
    }};
    ($($k:literal => $v:expr);+) => {{
        let mut q_vec: Vec<String> = Vec::new();
        $(
            q_vec.push($v.to_qs($k));
        )*
        q_vec.join("&")
    }};
}


#[cfg(test)]
mod tests {
    use std::collections::BTreeMap;
    use crate::qs::QueryString;

    #[test]
    fn test_stringify_common() {
        let x = 1;
        let y: Option<i32> = None;
        let s = stringify!{
            "a" => x;
            "b" => 2usize;
            "c" => 3f64;
            "d" => x + 6;
            "e" => vec![4i32, 5i32, 6i32];
            "f" => "hello world";
            "g" => Some(8);
            "h" => y
        };
        println!("{}", s);
        assert_eq!(s,  "a=1&b=2&c=3&d=7&e[]=4&e[]=5&e[]=6&f=hello%20world&g=8&h=");
    }

    #[test]
    fn test_stringify_map() {
        let mut m = BTreeMap::new();
        m.insert("a", 1);
        m.insert("b", 2);
        m.insert("c", 3);
        let s = stringify!(m);
        assert_eq!(s,  "a=1&b=2&c=3");

        let mut m = BTreeMap::new();
        m.insert("a", "hello world");
        m.insert("b", "ksd");
        m.insert("c", "turbidsoul");
        let s = stringify!(m);
        assert_eq!(s,  "a=hello%20world&b=ksd&c=turbidsoul");
    }

    #[test]
    fn test_stringify_tuple() {
        let x = 1;
        let s = stringify![
            ("a", x),
            ("b", 2i32),
            ("c", 3i32),
            ("d", x + 6),
            ("e", vec![4i32, 5i32, 6i32]),
            ("f", "hello world")
        ];
        assert_eq!(s,  "a=1&b=2&c=3&d=7&e[]=4&e[]=5&e[]=6&f=hello%20world");
    }
}
