//!
//! ```
//! let s = stringify!(
//!     "a" => 1,
//!     "b" => 2,
//!     "c" => 3
//! );
//! assert_eq!(s,  "a=1&b=2&c=3");
//!
//! let mut m = BTreeMap::new();
//! m.insert("a", 1);
//! m.insert("b", 2);
//! m.insert("c", 3);
//! let s = stringify!(m);
//! assert_eq!(s,  "a=1&b=2&c=3");
//!
//! let s = stringify![
//!     ("a", 1),
//!     ("b", 2),
//!     ("c", 3)
//! ];
//! assert_eq!(s,  "a=1&b=2&c=3");
//! ```
//! 


#[macro_export]
macro_rules! stringify {
    ($(($k:expr, $v:expr)),*) => {{
        let mut s = String::new();
        $(
            if s.len() > 0 {
                s.push('&');
            }
            s.push_str(&format!("{}={}", $k, $v).to_string());
        )*
        s
    }};
    ($m:expr) => {{
        let mut s = String::new();
        for (k, v) in $m.iter() {
            if s.len() > 0 {
                s.push('&');
            }
            s.push_str(&format!("{}={}", k.to_string(), v.to_string()).to_string());
        }
        s
    }};
    ($($k:expr => $v:expr),*) => {{
        let mut s = String::new();
        $(
            if s.len() > 0 {
                s.push('&');
            }
            s.push_str(&format!("{}={}", $k, $v).to_string());
        )*
        s
    }};
}



#[cfg(test)]
mod tests {
    use std::collections::BTreeMap;

    #[test]
    fn test_stringify_common() {
        let s = stringify!(
            "a" => 1,
            "b" => 2,
            "c" => 3
        );
        assert_eq!(s,  "a=1&b=2&c=3");
    }

    #[test]
    fn test_stringify_map() {
        let mut m = BTreeMap::new();
        m.insert("a", 1);
        m.insert("b", 2);
        m.insert("c", 3);
        let s = stringify!(m);
        assert_eq!(s,  "a=1&b=2&c=3");
    }

    #[test]
    fn test_stringify_tuple() {
        let s = stringify![
            ("a", 1),
            ("b", 2),
            ("c", 3)
        ];
        assert_eq!(s,  "a=1&b=2&c=3");
    }
}
