#[cfg(test)]
mod tests {
    #[test]
    fn get_clipboard_dummy_test() {
        assert!(crate::qrquick::get_clipboard().is_ascii())
    }
}

pub mod qrquick {
    use clipboard::ClipboardProvider;
    use clipboard::ClipboardContext;
    use qrcodegen::QrCode;
    use qrcodegen::QrCodeEcc;
    use png::Encoder;

    use std::path::Path;
    use std::error::Error;
    use std::fs::File;
    use std::io::BufWriter;
    use std::process::{Command, Stdio, Child};

    pub fn get_clipboard() -> String {
        let mut ctx: ClipboardContext = ClipboardProvider::new().unwrap();
        ctx.get_contents().unwrap()
    }
  
    pub fn set_clipboard() -> Result<Child, std::io::Error> {
        Command::new("xclip")
            .arg("-selection")
            .arg(r"clipboard")
            .arg("-target")
            .arg("image/png")
            .arg("-i")
            .arg("/tmp/qrcode.png")
            .stderr(Stdio::null())
            .stdin(Stdio::null())
            .stdout(Stdio::null())
            .spawn()
    }

    pub fn open_in_xdg() -> Result<Child, std::io::Error> {
        Command::new("xdg-open")
            .arg("/tmp/qrcode.png")
            .stderr(Stdio::null())
            .stdin(Stdio::null())
            .stdout(Stdio::null())
            .spawn()
    }

    pub fn make_qr_code(input: String) -> QrCode {
        match QrCode::encode_binary(input.as_bytes(), QrCodeEcc::Low) {
            Ok(img) => img,
            Err(e) => panic!("{}", e),
        }
    }
    
    pub fn qr_code_to_png(input: QrCode) -> Result<&'static Path, Box<dyn Error>> {
        let path = Path::new(r"/tmp/qrcode.png");
        let file = File::create(path)?;
        // Not adding this as ref mut fails for... some reason.
        #[allow(clippy::toplevel_ref_arg)]
        let ref mut w = BufWriter::new(file);
        let size: i32 = input.size();

        let mut encoder = Encoder::new(
            w, 
            (size * 2).try_into().unwrap(), 
            (size * 2).try_into().unwrap()
        );
        encoder.set_color(png::ColorType::Grayscale);

        let mut writer = encoder.write_header().unwrap();
        let mut data: Vec<u8> = Vec::new(); 
        for x in 0..size {
            for y in 0..size {
                data.push(
                    match input.get_module(x, y) {
                        true => 0,
                        false => 255
                    }
                );
                data.push(
                    match input.get_module(x, y) {
                        true => 0,
                        false => 255
                    }
                );
            }
            for y in 0..size {
                data.push(
                    match input.get_module(x, y) {
                        true => 0,
                        false => 255
                    }
                );
                data.push(
                    match input.get_module(x, y) {
                        true => 0,
                        false => 255
                    }
                );
            }
        }

        writer.write_image_data(&data)?;
        writer.finish()?;
        
        Ok(path)
    }
    
    pub fn run() {
        qr_code_to_png(make_qr_code(get_clipboard()))
            .expect("Failure while writing QR Code to /tmp/qrcode.png.");
        if let Err(e) = set_clipboard() {
            println!("Couldn't copy image to clipboard! {}\nOpening dialog...", e);
            open_in_xdg()
                .expect("Could not use xdg-open! QRCode is in /tmp/qrcode.png");
        }
    }
}

