/* Copyright (C) 2018 Olivier Goffart <ogoffart@woboq.com>

Permission is hereby granted, free of charge, to any person obtaining a copy of this software and
associated documentation files (the "Software"), to deal in the Software without restriction,
including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense,
and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so,
subject to the following conditions:

The above copyright notice and this permission notice shall be included in all copies or substantial
portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT
NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES
OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/
use std::collections::BTreeMap;
use std::env;
use std::fs;
use std::path::PathBuf;

use proc_macro::TokenStream;
use quote::quote;
use syn::parse::{Parse, ParseStream, Result};
use syn::{braced, parse_macro_input, Ident, LitStr, Token, Visibility};

/// Function with this name and visibility modifier will be generated by the `qrc!` macro.
struct TargetFunc {
    vis: Visibility,
    ident: Ident,
}

impl Parse for TargetFunc {
    fn parse(input: ParseStream) -> Result<Self> {
        let vis = input.parse()?;
        let ident = input.parse()?;
        Ok(TargetFunc { vis, ident })
    }
}

/// ```txt
/// Resource   ::= $( $base_dir:physical as )? $prefix:virtual { $( $f:File ),* }
/// ```
#[derive(Debug)]
struct Resource {
    /// Local file system paths in this resource are relative to this base directory.
    base_dir: Option<String>,
    /// Virtual file system paths in qrc are prefixed with this path.
    prefix: String,
    /// Vector of files inside this resource.
    ///
    /// Their aliases (virtual paths) will be prefixed by the `prefix` path,
    /// and their location on the local file system will be prefixed by `base_dir`.
    files: Vec<File>,
}

impl Resource {
    ///  - **Note** about physical path: _resource_'s base directory is prepended to
    ///    the file's physical path before looking for the file on the local file
    ///    system, but after the physical path is cloned to the virtual counterpart
    ///    (if the later one was omitted, i.e. no explicit alias was given).
    pub fn files<'a>(&'a self) -> impl Iterator<Item = File> + 'a {
        self.files.iter().map(move |f| self.resolve(f))
    }

    fn resolve(&self, file: &File) -> File {
        match self.base_dir.as_ref() {
            Some(base_dir) => {
                let f = File {
                    path: format!("./{}/{}", base_dir, file.path.clone()),
                    alias: Some(file.resolved_alias().to_owned()),
                };
                f
            }
            None => file.clone(),
        }
    }
}

impl Parse for Resource {
    fn parse(input: ParseStream) -> Result<Self> {
        // Mutable horror of grammars with optional prefix
        let mut base_dir = None;
        let mut prefix = input.parse::<LitStr>()?.value();
        if let Some(_) = input.parse::<Option<Token![as]>>()? {
            base_dir = Some(prefix);
            prefix = input.parse::<LitStr>()?.value();
        }
        let files = {
            let content;
            braced!(content in input);
            content.parse_terminated::<File, Token![,]>(File::parse)?.into_iter().collect()
        };
        Ok(Resource { base_dir, prefix, files })
    }
}

/// ```txt
/// File       ::= $path:physical $( as $alias:virtual )?
/// ```
#[derive(Clone, Debug)]
struct File {
    /// Physical path on local file system
    path: String,
    /// Virtual path in qrc:/// file system
    alias: Option<String>,
}

impl Parse for File {
    fn parse(input: ParseStream) -> Result<Self> {
        let file = input.parse::<LitStr>()?.value();
        let mut alias = None;
        if let Some(_) = input.parse::<Option<Token![as]>>()? {
            alias = Some(input.parse::<LitStr>()?.value());
        }
        Ok(File { path: file, alias })
    }
}

impl File {
    // use alias, if one was explicitly provided, else physical path.
    pub fn resolved_alias(&self) -> &str {
        &*self.alias.as_ref().unwrap_or(&self.path)
    }
}

struct QrcMacro {
    func: TargetFunc,
    data: Vec<Resource>,
}

impl Parse for QrcMacro {
    fn parse(input: ParseStream) -> Result<Self> {
        let func = input.parse()?;
        input.parse::<Option<Token![,]>>()?;
        // Rest of tokens are `Resource`s separated by comma
        let data =
            input.parse_terminated::<Resource, Token![,]>(Resource::parse)?.into_iter().collect();
        Ok(QrcMacro { func, data })
    }
}

fn qt_hash(key: &str) -> u32 {
    let mut h = 0u32;

    for p in key.chars() {
        assert_eq!(p.len_utf16(), 1, "Surrogate pair not supported by the hash function");
        h = (h << 4) + p as u32;
        h ^= (h & 0xf0000000) >> 23;
        h &= 0x0fffffff;
    }
    h
}

/// Special string class used to represent Qt strings with pre-computed stable hash.
///
/// Here it is used to represent paths (directories and files) in qrc virtual file system.
#[derive(Debug, Eq, PartialEq, PartialOrd, Ord, Clone)]
struct HashedString {
    hash: u32,
    string: String,
}
impl HashedString {
    fn new(string: String) -> HashedString {
        let hash = qt_hash(&string);
        HashedString { hash, string }
    }
}

/// Virtual file system tree, where leafs (files) are pointing to
/// real files on the local file system.
#[derive(Debug)]
enum TreeNode {
    /// Path to a file on a local file system.
    File(String),
    /// Content of a directory for a qrc virtual file system.
    ///
    /// Directory doesn't know its own name, but rather its parent directory knows.
    ///
    /// Maps qrc virtual paths onto other virtual directories and eventually
    /// onto files on the local file system.
    Directory { contents: BTreeMap<HashedString, TreeNode>, offset: u32 },
}
impl TreeNode {
    /// Create new directory on the qrc virtual file system.
    fn new_dir() -> TreeNode {
        TreeNode::Directory { contents: Default::default(), offset: 0 }
    }

    /// Create new reference to the path on the local file system.
    fn new_file(file: String) -> TreeNode {
        TreeNode::File(file)
    }

    /// `virtual_rel_path` is a path in qrc virtual file system, relative to the `node`.
    fn insert_node(&mut self, virtual_rel_path: &str, node: TreeNode) {
        let contents = match self {
            TreeNode::Directory { contents, .. } => contents,
            _ => panic!("root not a dir?"),
        };

        if virtual_rel_path == "" {
            // insert into itself
            contents.extend(match node {
                TreeNode::Directory { contents, .. } => contents,
                _ => panic!("merge file and directory?"),
            });
            return;
        }

        match virtual_rel_path.find('/') {
            Some(idx) => {
                let (name, rest) = virtual_rel_path.split_at(idx);
                let hashed = HashedString::new(name.into());
                contents
                    .entry(hashed)
                    .or_insert_with(TreeNode::new_dir)
                    .insert_node(&rest[1..], node);
            }
            None => {
                let hashed = HashedString::new(virtual_rel_path.into());
                contents
                    .insert(hashed, node)
                    .and_then(|_| -> Option<()> { panic!("Several time the same file?") });
            }
        };
    }

    fn compute_offsets(&mut self, mut offset: u32) -> u32 {
        if let TreeNode::Directory { contents, offset: o } = self {
            *o = offset;
            offset += contents.len() as u32;
            for node in contents.values_mut() {
                offset = node.compute_offsets(offset);
            }
        }
        offset
    }
}

// remove duplicate, or leading '/'
fn simplify_prefix(mut s: String) -> String {
    let mut last_slash = true; // so we remove the first '/'
    s.retain(|x| {
        let r = last_slash && x == '/';
        last_slash = x == '/';
        !r
    });
    if last_slash {
        s.pop();
    }
    s
}

#[test]
fn simplify_prefix_test() {
    assert_eq!(simplify_prefix("/".into()), "");
    assert_eq!(simplify_prefix("///".into()), "");
    assert_eq!(simplify_prefix("/foo//bar/d".into()), "foo/bar/d");
    assert_eq!(simplify_prefix("hello/".into()), "hello");
}

fn build_tree(resources: Vec<Resource>) -> TreeNode {
    let mut root = TreeNode::new_dir();
    for r in resources {
        let mut node = TreeNode::new_dir();
        for f in r.files() {
            let local_file = TreeNode::new_file(f.path.clone());
            let virt_path = f.resolved_alias().to_owned();
            node.insert_node(&*virt_path, local_file);
        }
        root.insert_node(&simplify_prefix(r.prefix), node);
    }
    root
}

fn push_u32_be(v: &mut Vec<u8>, val: u32) {
    v.extend_from_slice(&[
        ((val >> 24) & 0xff) as u8,
        ((val >> 16) & 0xff) as u8,
        ((val >> 8) & 0xff) as u8,
        (val & 0xff) as u8,
    ]);
}

fn push_u16_be(v: &mut Vec<u8>, val: u16) {
    v.extend_from_slice(&[((val >> 8) & 0xff) as u8, (val & 0xff) as u8]);
}

#[derive(Default, Debug)]
struct Data {
    payload: Vec<u8>,
    names: Vec<u8>,
    tree_data: Vec<u8>,
    files: Vec<String>,
}
impl Data {
    fn insert_file(&mut self, filename: &str) {
        let mut filepath = PathBuf::new();
        if let Ok(cargo_manifest) = env::var("CARGO_MANIFEST_DIR") {
            filepath.push(cargo_manifest);
        }

        filepath.push(filename);

        let mut data = fs::read(&filepath)
            .unwrap_or_else(|_| panic!("Cannot open file {}", filepath.display()));
        push_u32_be(&mut self.payload, data.len() as u32);
        self.payload.append(&mut data);
        self.files.push(filepath.to_str().expect("File path contains invalid Unicode").into());
    }

    fn insert_directory(&mut self, contents: &BTreeMap<HashedString, TreeNode>) {
        for (ref name, ref val) in contents {
            let name_off = self.insert_name(name);
            push_u32_be(&mut self.tree_data, name_off);
            match val {
                TreeNode::File(ref filename) => {
                    push_u16_be(&mut self.tree_data, 0); // flags
                    push_u16_be(&mut self.tree_data, 0); // country
                    push_u16_be(&mut self.tree_data, 1); // lang (C)
                    let offset = self.payload.len();
                    push_u32_be(&mut self.tree_data, offset as u32);
                    self.insert_file(filename);
                }
                TreeNode::Directory { ref contents, offset } => {
                    push_u16_be(&mut self.tree_data, 2); // directory flag
                    push_u32_be(&mut self.tree_data, contents.len() as u32);
                    push_u32_be(&mut self.tree_data, *offset);
                }
            }
            // modification time (64 bit) FIXME
            push_u32_be(&mut self.tree_data, 0);
            push_u32_be(&mut self.tree_data, 0);
        }
        for val in contents.values() {
            if let TreeNode::Directory { ref contents, .. } = val {
                self.insert_directory(contents)
            }
        }
    }

    fn insert_name(&mut self, name: &HashedString) -> u32 {
        let offset = self.names.len();
        push_u16_be(&mut self.names, name.string.len() as u16);
        push_u32_be(&mut self.names, name.hash);

        for p in name.string.chars() {
            assert_eq!(p.len_utf16(), 1, "Surrogate pair not supported");
            push_u16_be(&mut self.names, p as u16);
        }
        offset as u32
    }
}

fn generate_data(root: &TreeNode) -> Data {
    let mut d = Data::default();

    let contents = match root {
        TreeNode::Directory { ref contents, .. } => contents,
        _ => panic!("root not a dir?"),
    };

    // first item
    push_u32_be(&mut d.tree_data, 0); // fake name
    push_u16_be(&mut d.tree_data, 2); // flag
    push_u32_be(&mut d.tree_data, contents.len() as u32);
    push_u32_be(&mut d.tree_data, 1); // first offset

    // modification time (64 bit) FIXME
    push_u32_be(&mut d.tree_data, 0);
    push_u32_be(&mut d.tree_data, 0);

    d.insert_directory(contents);
    d
}

fn expand_macro(func: TargetFunc, data: Data) -> TokenStream {
    let TargetFunc { vis, ident } = func;
    let Data { payload, names, tree_data, files } = data;

    // This is the fastest way to include a byte slice.
    let payload =
        ::proc_macro2::TokenTree::Literal(::proc_macro2::Literal::byte_string(payload.as_slice()));

    let q = quote! {
        #vis fn #ident() {
            use ::std::sync::Once;
            static INIT_RESOURCES: Once = Once::new();
            INIT_RESOURCES.call_once(|| {
                static PAYLOAD : &'static [u8] = #payload;
                static NAMES : &'static [u8] = & [ #(#names),* ];
                static TREE_DATA : &'static [u8] = & [ #(#tree_data),* ];
                unsafe { ::qmetaobject::qrc::register_resource_data(2, TREE_DATA, NAMES, PAYLOAD) };
                // Because we want that the macro re-compiles if the contents of the file changes!
                #({ const _X: &'static [ u8 ] = include_bytes!(#files); })*
            });
        }
    };
    q.into()
}

pub fn process_qrc(source: TokenStream) -> TokenStream {
    let parsed = parse_macro_input!(source as QrcMacro);
    let mut tree = build_tree(parsed.data);
    tree.compute_offsets(1);
    let d = generate_data(&tree);
    expand_macro(parsed.func, d)
}
