use std::cmp::Ordering;
use std::collections::VecDeque;

use crate::common_refactor::*;
use crate::*;

// Simple wrapper so we can implement the `OrderedContainer` trait
// without naming conflict with `VecDeque<_>`.
#[derive(Debug, Eq, PartialEq)]
pub(crate) struct UnitBucket<O: Ord, K: Eq>(pub VecDeque<OPair<O, K>>);

impl<O: Ord, K: Eq> OrderedContainer<O, K> for UnitBucket<O, K> {
    fn oc_new() -> Self {
        Self(VecDeque::new())
    }

    fn oc_front(&self) -> Option<&OPair<O, K>> {
        self.0.front()
    }

    fn oc_back(&self) -> Option<&OPair<O, K>> {
        self.0.back()
    }

    fn oc_cmp(&self, o: &O) -> Option<Ordering> {
        if self.0.len() == 0 {
            None
        } else {
            cmp_to_non_empty(self, o)
        }
    }

    fn oc_push_front(&mut self, op: OPair<O, K>) -> Result<(), OCFull<OPair<O, K>>> {
        if self.0.len() < MAX_NUM_SUBITEMS {
            self.0.push_front(op);
            Ok(())
        } else {
            Err(OCFull(op))
        }
    }

    fn oc_push_back(&mut self, op: OPair<O, K>) -> Result<(), OCFull<OPair<O, K>>> {
        if self.0.len() < MAX_NUM_SUBITEMS {
            self.0.push_back(op);
            Ok(())
        } else {
            Err(OCFull(op))
        }
    }

    fn oc_pop_front(&mut self) -> Option<OPair<O, K>> {
        self.0.pop_front()
    }

    fn oc_pop_back(&mut self) -> Option<OPair<O, K>> {
        self.0.pop_back()
    }

    fn oc_split_off(&mut self, i: usize) -> Self {
        Self(self.0.split_off(i))
    }

    fn oc_binary_search(&self, o: &O) -> Result<usize, usize> {
        self.0.binary_search_by(|el| el.pos.cmp(o))
    }

    fn oc_len(&self) -> usize {
        self.0.len()
    }

    fn insert_at_split(&mut self, ins: OPair<O, K>) -> Self {
        let split_index = match self.oc_binary_search(&ins.pos) {
            Ok(i) => i,
            Err(i) => i,
        };
        let mut right = self.oc_split_off(split_index);
        if self.0.len() <= right.0.len() {
            self.0.push_back(ins);
        } else {
            right.0.push_front(ins);
        }
        right
    }

    fn insert_sort(&mut self, ins: OPair<O, K>) -> Result<(), OCFull<OPair<O, K>>> {
        if self.0.len() >= MAX_NUM_SUBITEMS {
            return Err(OCFull(ins));
        }
        let insert_index = match self.oc_binary_search(&ins.pos) {
            Ok(i) => i,
            Err(i) => i,
        };
        self.0.insert(insert_index, ins);
        Ok(())
    }

    fn remove_at_pos_if<F>(&mut self, cmp: &O, key_criteria: F) -> Vec<OPair<O, K>>
    where
        F: Fn(&OPair<O, K>) -> bool,
    {
        let mut indexes_to_remove = Vec::new();

        if let Ok(index_found) = self.oc_binary_search(cmp) {
            if key_criteria(&self.0[index_found]) {
                indexes_to_remove.push(index_found);
            }
            if index_found != 0 {
                let mut left = index_found - 1;
                while cmp == &self.0[left].pos {
                    if key_criteria(&self.0[left]) {
                        indexes_to_remove.push(left);
                    }
                    if left == 0 {
                        break;
                    }
                    left -= 1;
                }
            };
            {
                let mut right = index_found + 1;
                let len = self.0.len();
                while right < len && cmp == &self.0[right].pos {
                    if key_criteria(&self.0[right]) {
                        indexes_to_remove.push(right);
                    }
                    right += 1;
                }
            };
        };
        del_multi(&mut self.0, indexes_to_remove.as_ref())
    }

    #[cfg(test)]
    fn recursive_total_len(&self) -> usize {
        self.0.len()
    }
}

// Mainly for testing, no sorted order check.
impl<O: Ord, K: Eq> From<Vec<OPair<O, K>>> for UnitBucket<O, K> {
    fn from(vec: Vec<OPair<O, K>>) -> Self {
        Self(vec.into())
    }
}

#[cfg(test)]
mod tests {
    use super::*;
    use crate::test_utils::*;

    #[test]
    fn unit_bucket_simple_ops_sans_remove() {
        let mut num_bucket = UnitBucket::oc_new();
        assert!(num_bucket.oc_pop_back().is_none());
        assert!(num_bucket.oc_pop_front().is_none());
        assert_eq!(None, num_bucket.oc_cmp(&556));
        let k = "meh";

        assert!(num_bucket.insert_sort(OPair { pos: 124, key: k }).is_ok());
        assert!(num_bucket.insert_sort(OPair { pos: 191, key: k }).is_ok());
        assert!(num_bucket.insert_sort(OPair { pos: 2, key: k }).is_ok());
        assert!(num_bucket.insert_sort(OPair { pos: 56, key: k }).is_ok());
        assert!(num_bucket.insert_sort(OPair { pos: 33, key: k }).is_ok());
        assert!(num_bucket.insert_sort(OPair { pos: 11, key: k }).is_ok());
        assert!(num_bucket.insert_sort(OPair { pos: 1009, key: k }).is_ok());
        assert!(num_bucket.insert_sort(OPair { pos: 333, key: k }).is_ok());
        assert!(num_bucket.insert_sort(OPair { pos: 33, key: k }).is_ok());
        assert!(num_bucket.insert_sort(OPair { pos: 11, key: k }).is_ok());
        assert!(num_bucket.insert_sort(OPair { pos: 124, key: k }).is_ok());
        assert!(num_bucket.insert_sort(OPair { pos: 1009, key: k }).is_ok());
        assert!(num_bucket.insert_sort(OPair { pos: 1008, key: k }).is_ok());
        assert_eq!(
            num_bucket.0,
            vec![
                OPair { pos: 2, key: k },
                OPair { pos: 11, key: k },
                OPair { pos: 11, key: k },
                OPair { pos: 33, key: k },
                OPair { pos: 33, key: k },
                OPair { pos: 56, key: k },
                OPair { pos: 124, key: k },
                OPair { pos: 124, key: k },
                OPair { pos: 191, key: k },
                OPair { pos: 333, key: k },
                OPair { pos: 1008, key: k },
                OPair { pos: 1009, key: k },
                OPair { pos: 1009, key: k }
            ]
        );
        assert_eq!(Ordering::Equal, num_bucket.oc_cmp(&556).unwrap());
        assert_eq!(Ordering::Equal, num_bucket.oc_cmp(&1009).unwrap());
        assert_eq!(Ordering::Equal, num_bucket.oc_cmp(&2).unwrap());
        assert_eq!(Ordering::Greater, num_bucket.oc_cmp(&1).unwrap());
        assert_eq!(Ordering::Less, num_bucket.oc_cmp(&1010).unwrap());
        assert_eq!(2, num_bucket.oc_pop_front().unwrap().pos);
        assert_eq!(1009, num_bucket.oc_pop_back().unwrap().pos);
        assert_eq!(11, num_bucket.0.len());
        assert!(num_bucket.oc_push_back(OPair { pos: 2000, key: k }).is_ok());
        assert!(num_bucket.oc_push_front(OPair { pos: 11, key: k }).is_ok());
        assert!(num_bucket.oc_push_front(OPair { pos: 9, key: k }).is_ok());

        // fill up to capacity
        for _ in 0..MAX_NUM_SUBITEMS - num_bucket.0.len() {
            assert!(num_bucket.insert_sort(OPair { pos: 0, key: k }).is_ok());
        }
        let insert_err = num_bucket.insert_sort(OPair { pos: 0, key: k });
        assert_eq!(insert_err, Err(OCFull(OPair { pos: 0, key: k })));

        let push_front_err = num_bucket.oc_push_front(OPair { pos: 0, key: k });
        assert_eq!(Err(OCFull(OPair { pos: 0, key: k })), push_front_err);

        let push_back_err = num_bucket.oc_push_back(OPair { pos: 3000, key: k });
        assert_eq!(Err(OCFull(OPair { pos: 3000, key: k })), push_back_err);

        let mut to_split = UnitBucket(VecDeque::from(vec![
            OPair { pos: 2, key: k },
            OPair { pos: 4, key: k },
            OPair { pos: 6, key: k },
            OPair { pos: 8, key: k },
            OPair { pos: 10, key: k },
            OPair { pos: 12, key: k },
            OPair { pos: 14, key: k },
            OPair { pos: 16, key: k },
            OPair { pos: 18, key: k },
            OPair { pos: 20, key: k },
        ]));
        let right = to_split.insert_at_split(OPair { pos: 19, key: k });
        assert_eq!(
            to_split.0,
            vec![
                OPair { pos: 2, key: k },
                OPair { pos: 4, key: k },
                OPair { pos: 6, key: k },
                OPair { pos: 8, key: k },
                OPair { pos: 10, key: k },
                OPair { pos: 12, key: k },
                OPair { pos: 14, key: k },
                OPair { pos: 16, key: k },
                OPair { pos: 18, key: k },
            ]
        );
        assert_eq!(
            right.0,
            vec![OPair { pos: 19, key: k }, OPair { pos: 20, key: k },]
        );
        let right = to_split.insert_at_split(OPair { pos: 19, key: k });
        assert_eq!(
            to_split.0,
            vec![
                OPair { pos: 2, key: k },
                OPair { pos: 4, key: k },
                OPair { pos: 6, key: k },
                OPair { pos: 8, key: k },
                OPair { pos: 10, key: k },
                OPair { pos: 12, key: k },
                OPair { pos: 14, key: k },
                OPair { pos: 16, key: k },
                OPair { pos: 18, key: k },
            ]
        );
        assert_eq!(right.0, vec![OPair { pos: 19, key: k }]);
        let right = to_split.insert_at_split(OPair { pos: 3, key: k });
        assert_eq!(
            to_split.0,
            vec![OPair { pos: 2, key: k }, OPair { pos: 3, key: k }]
        );
        assert_eq!(
            right.0,
            vec![
                OPair { pos: 4, key: k },
                OPair { pos: 6, key: k },
                OPair { pos: 8, key: k },
                OPair { pos: 10, key: k },
                OPair { pos: 12, key: k },
                OPair { pos: 14, key: k },
                OPair { pos: 16, key: k },
                OPair { pos: 18, key: k },
            ]
        );
        let right = to_split.insert_at_split(OPair { pos: 0, key: k });
        assert_eq!(to_split.0, vec![OPair { pos: 0, key: k }]);
        assert_eq!(
            right.0,
            vec![OPair { pos: 2, key: k }, OPair { pos: 3, key: k }]
        );
    }

    #[test]
    fn ub_insert_sort_until_full() {
        let k = "meh";
        let mut num_bucket = UnitBucket::oc_new();
        let randoms = random_vec::<usize>(MAX_NUM_SUBITEMS);
        for o in randoms {
            assert!(num_bucket.insert_sort(OPair { pos: o, key: k }).is_ok());
        }
        assert_eq!(
            Err(OCFull(OPair {
                pos: 32169874,
                key: k
            })),
            num_bucket.insert_sort(OPair {
                pos: 32169874,
                key: k
            })
        );
        for i in 0..num_bucket.0.len() {
            if i == num_bucket.0.len() - 1 {
                break;
            };
            assert!(num_bucket.0[i] <= num_bucket.0[i + 1]);
        }
    }

    #[test]
    fn unit_bucket_remove() {
        let mut tup_bucket: UnitBucket<i32, char> = UnitBucket::oc_new();
        assert!(tup_bucket.insert_sort(OPair { pos: 319, key: 'd' }).is_ok());
        assert!(tup_bucket.insert_sort(OPair { pos: 319, key: 't' }).is_ok());
        assert!(tup_bucket.insert_sort(OPair { pos: 201, key: 'a' }).is_ok());
        assert!(tup_bucket.insert_sort(OPair { pos: 202, key: 't' }).is_ok());
        assert!(tup_bucket.insert_sort(OPair { pos: 200, key: 't' }).is_ok());
        assert!(tup_bucket.insert_sort(OPair { pos: 200, key: 'c' }).is_ok());
        assert!(tup_bucket.insert_sort(OPair { pos: 200, key: 'n' }).is_ok());
        assert!(tup_bucket.insert_sort(OPair { pos: 200, key: 'n' }).is_ok());
        assert!(tup_bucket.insert_sort(OPair { pos: 200, key: 'n' }).is_ok());

        let rem = tup_bucket.remove_at_pos_if(&200, |pair| pair.key >= 'n');
        assert_eq!(4, rem.len());
        assert_eq!(5, tup_bucket.0.len());
    }
}
