use rand::distributions::Standard;
use rand::prelude::*;
use std::collections::VecDeque;

use data_encoding::BASE64URL_NOPAD;

use crate::unit_bucket::UnitBucket;
use crate::*;

pub(crate) fn random_vec<T>(num: usize) -> Vec<T>
where
    Standard: Distribution<T>,
    T: Ord,
{
    let mut data = Vec::new();
    let mut rng = rand::thread_rng();
    for _ in 0..num {
        data.push(rng.gen())
    }
    data
}

pub(crate) fn is_sorted<O: Ord>(v: &VecDeque<O>) -> bool {
    for i in 0..v.len() - 1 {
        if v[i] > v[i + 1] {
            return false;
        }
    }
    true
}

pub(crate) fn nested_is_sorted<O: Ord, K: Eq>(v: &VecDeque<UnitBucket<O, K>>) -> bool {
    for i in 0..v.len() - 1 {
        if !is_sorted(&v[i].0) {
            return false;
        }
        if v[i].0.back().unwrap() > v[i + 1].0.front().unwrap() {
            return false;
        }
    }
    true
}

pub(crate) fn double_nested_serial() -> VecDeque<VecDeque<UnitBucket<usize, &'static str>>> {
    let k = "meh";
    let mut double_nested_bucket: VecDeque<VecDeque<UnitBucket<usize, &str>>> = VecDeque::new();
    for i in 0..MAX_NUM_SUBITEMS ^ 3 {
        assert!(double_nested_bucket
            .oc_push_back(OPair { pos: i, key: k })
            .is_ok());
    }
    double_nested_bucket
}

pub(crate) fn deep_total<O: Ord, K: Eq>(v: &VecDeque<UnitBucket<O, K>>) -> usize {
    let mut total = 0;
    for ub in v {
        total += ub.0.len();
    }
    total
}

pub(crate) fn roughly_within(target: usize, mid: usize, tolerance_percent: usize) -> bool {
    let lower_limit = mid * (100 - tolerance_percent) / 100;
    let upper_limit = mid * (100 + tolerance_percent) / 100;
    target >= lower_limit && target <= upper_limit
}

pub(crate) fn base64(input: &[u8]) -> String {
    let mut out = String::new();
    BASE64URL_NOPAD.encode_append(input, &mut out);
    out
}

pub(crate) fn base64_int(input: i32) -> String {
    base64(input.to_le_bytes().as_ref())
}

#[test]
fn test_is_sorted() {
    assert!(is_sorted(&VecDeque::from(vec![1, 6, 6, 7, 8, 9, 9, 100])));
    assert!(!is_sorted(&VecDeque::from(vec![1, 6, 6, 7, 8, 9, 100, 99])));
}

#[test]
fn test_nested_is_sorted() {
    let k = "meh";
    assert!(nested_is_sorted(
        &vec![
            vec![
                OPair { pos: 4, key: k },
                OPair { pos: 5, key: k },
                OPair { pos: 6, key: k },
                OPair { pos: 6, key: k },
            ]
            .into(),
            vec![
                OPair { pos: 6, key: k },
                OPair { pos: 6, key: k },
                OPair { pos: 6, key: k },
                OPair { pos: 6, key: k },
            ]
            .into(),
            vec![
                OPair { pos: 6, key: k },
                OPair { pos: 7, key: k },
                OPair { pos: 8, key: k },
                OPair { pos: 9, key: k },
            ]
            .into(),
        ]
        .into()
    ));
    assert!(!nested_is_sorted(
        &vec![
            vec![
                OPair { pos: 4, key: k },
                OPair { pos: 5, key: k },
                OPair { pos: 6, key: k },
                OPair { pos: 6, key: k },
            ]
            .into(),
            vec![
                OPair { pos: 6, key: k },
                OPair { pos: 7, key: k },
                OPair { pos: 8, key: k },
                OPair { pos: 9, key: k },
            ]
            .into(),
            vec![
                OPair { pos: 6, key: k },
                OPair { pos: 6, key: k },
                OPair { pos: 6, key: k },
                OPair { pos: 6, key: k },
            ]
            .into(),
        ]
        .into()
    ));
}

#[test]
fn test_roughly_within() {
    assert!(roughly_within(54, 50, 10));
    assert!(roughly_within(46, 50, 10));
    assert!(!roughly_within(56, 50, 10));
    assert!(!roughly_within(44, 50, 10));
}

#[ignore]
#[test]
fn test_double_nested_serial() {
    let k = "meh";
    let mut double_nested_bucket = double_nested_serial();
    let num = MAX_NUM_SUBITEMS * MAX_NUM_SUBITEMS * MAX_NUM_SUBITEMS;
    assert_eq!(
        Err(OCFull(OPair { pos: num, key: k })),
        double_nested_bucket.oc_push_back(OPair { pos: num, key: k })
    );
}
