use std::cmp::Ordering;
use std::collections::VecDeque;

use crate::common_refactor::*;
use crate::unit_bucket::UnitBucket;
use crate::*;

#[derive(Debug)]
pub(crate) struct NestedBucket<O: Ord, K: Eq> {
    // Here we assume that we have designed inner to have enough
    // nesting levels such that we will run out of physical memory
    // before it will ever be full.
    // The error returning mechanism is meant for lower bucket to
    // signal that it's full.
    // Any push/insert to a full inner will result in a panic.
    // If a panic is ever encountered on any system, it means
    // inner has been designed with the wrong capacity.
    // Assume sparse population.
    inner: VecDeque<VecDeque<VecDeque<VecDeque<UnitBucket<O, K>>>>>,
}

impl<O: Ord, K: Eq> NestedBucket<O, K> {
    pub fn new() -> Self {
        Self {
            inner: VecDeque::new(),
        }
    }

    pub fn front(&self) -> Option<&OPair<O, K>> {
        self.inner.oc_front()
    }

    pub fn back(&self) -> Option<&OPair<O, K>> {
        self.inner.oc_back()
    }

    pub fn pop_front(&mut self) -> Option<OPair<O, K>> {
        self.inner.oc_pop_front()
    }

    pub fn pop_back(&mut self) -> Option<OPair<O, K>> {
        self.inner.oc_pop_back()
    }

    // No order check, perform beforehand.
    pub fn push_front(&mut self, pair: OPair<O, K>) {
        if let Err(OCFull(_)) = self.inner.oc_push_front(pair) {
            panic!("NestedBucket full before push_front");
        }
    }

    // No order check, perform beforehand.
    pub fn push_back(&mut self, pair: OPair<O, K>) {
        if let Err(OCFull(_)) = self.inner.oc_push_back(pair) {
            panic!("NestedBucket full before push_back");
        }
    }

    pub fn insert(&mut self, ins: OPair<O, K>) {
        if let Err(OCFull(_)) = self.inner.insert_sort(ins) {
            panic!("NestedBucket full before insert");
        }
    }

    pub fn remove<F>(&mut self, pos: &O, criteria: F) -> Vec<OPair<O, K>>
    where
        F: Fn(&OPair<O, K>) -> bool,
    {
        self.inner.remove_at_pos_if(pos, criteria)
    }

    #[cfg(test)]
    pub fn recursive_total_len(&self) -> usize {
        self.inner.recursive_total_len()
    }
}

impl<O, OC, K> OrderedContainer<O, K> for VecDeque<OC>
where
    O: Ord,
    OC: OrderedContainer<O, K>,
    K: Eq,
{
    fn oc_new() -> Self {
        VecDeque::new()
    }

    fn oc_front(&self) -> Option<&OPair<O, K>> {
        if let Some(first_element) = self.front() {
            first_element.oc_front()
        } else {
            None
        }
    }

    fn oc_back(&self) -> Option<&OPair<O, K>> {
        if let Some(last_element) = self.back() {
            last_element.oc_back()
        } else {
            None
        }
    }

    fn oc_cmp(&self, o: &O) -> Option<Ordering> {
        if self.len() == 0 {
            None
        } else {
            cmp_to_non_empty(self, o)
        }
    }

    fn oc_push_front(&mut self, o: OPair<O, K>) -> Result<(), OCFull<OPair<O, K>>> {
        let self_len = self.len();
        if self_len == 0 {
            self.push_front(OC::oc_new());
        };
        // unwrap safety: we just made sure self has at least 1 element
        match self.front_mut().unwrap().oc_push_front(o) {
            Ok(_) => Ok(()),
            Err(OCFull(o)) => {
                if self_len < MAX_NUM_SUBITEMS {
                    self.push_front(push_to_empty_new_subgroup(o));
                    Ok(())
                } else {
                    Err(OCFull(o))
                }
            }
        }
    }

    fn oc_push_back(&mut self, o: OPair<O, K>) -> Result<(), OCFull<OPair<O, K>>> {
        let self_len = self.len();
        if self_len == 0 {
            self.push_back(OC::oc_new());
        };
        // unwrap safety: we just made sure self has at least 1 element
        match self.back_mut().unwrap().oc_push_back(o) {
            Ok(_) => Ok(()),
            Err(OCFull(o)) => {
                if self_len < MAX_NUM_SUBITEMS {
                    self.push_back(push_to_empty_new_subgroup(o));
                    Ok(())
                } else {
                    Err(OCFull(o))
                }
            }
        }
    }

    fn oc_pop_front(&mut self) -> Option<OPair<O, K>> {
        if let Some(front) = self.front_mut() {
            let out = front.oc_pop_front();
            if front.oc_len() == 0 {
                self.pop_front();
            }
            out
        } else {
            None
        }
    }

    fn oc_pop_back(&mut self) -> Option<OPair<O, K>> {
        if let Some(back) = self.back_mut() {
            let out = back.oc_pop_back();
            if back.oc_len() == 0 {
                self.pop_back();
            }
            out
        } else {
            None
        }
    }

    fn oc_split_off(&mut self, i: usize) -> Self {
        self.split_off(i)
    }

    fn oc_binary_search(&self, o: &O) -> Result<usize, usize> {
        if self.len() == 0 {
            Err(0)
        } else {
            self.binary_search_by(|child| child.oc_cmp(o).unwrap())
        }
    }

    fn oc_len(&self) -> usize {
        self.len()
    }

    fn insert_at_split(&mut self, ins: OPair<O, K>) -> Self {
        match self.oc_binary_search(&ins.pos) {
            Ok(i) => {
                let right_inner = self[i].insert_at_split(ins);
                let mut right_outer = self.oc_split_off(i + 1);
                right_outer.push_front(right_inner);
                right_outer
            }
            Err(i) => {
                let mut right_outer = self.oc_split_off(i);
                if let Err(OCFull(ins)) = right_outer.oc_push_front(ins) {
                    let _ = self.oc_push_back(ins);
                    // one of them (left or right) can't be full
                };
                right_outer
            }
        }
    }

    fn insert_sort(&mut self, ins: OPair<O, K>) -> Result<(), OCFull<OPair<O, K>>> {
        let self_len = self.len();
        if self_len == 0 {
            let _ = self.oc_push_front(ins); // push to empty, can't fail
            return Ok(());
        };
        match self.oc_binary_search(&ins.pos) {
            Ok(i) => match self[i].insert_sort(ins) {
                Ok(()) => Ok(()),
                Err(OCFull(o)) => {
                    if self_len < MAX_NUM_SUBITEMS {
                        let right_of_split = self[i].insert_at_split(o);
                        self.insert(i + 1, right_of_split);
                        Ok(())
                    } else {
                        Err(OCFull(o))
                    }
                }
            },
            Err(i) => {
                if i == self_len {
                    if let Err(OCFull(pair)) = self.oc_push_back(ins) {
                        if self_len < MAX_NUM_SUBITEMS {
                            self.push_back(push_to_empty_new_subgroup(pair));
                            Ok(())
                        } else {
                            Err(OCFull(pair))
                        }
                    } else {
                        Ok(())
                    }
                } else if i == 0 {
                    if let Err(OCFull(pair)) = self.oc_push_front(ins) {
                        if self_len < MAX_NUM_SUBITEMS {
                            self.push_front(push_to_empty_new_subgroup(pair));
                            Ok(())
                        } else {
                            Err(OCFull(pair))
                        }
                    } else {
                        Ok(())
                    }
                } else {
                    if let Err(OCFull(pair)) = self[i].oc_push_front(ins) {
                        if i == 0 {
                            return new_subgroup_then_insert(self, i, pair);
                        }
                        if let Err(OCFull(pair)) = self[i - 1].oc_push_back(pair) {
                            new_subgroup_then_insert(self, i, pair)
                        } else {
                            Ok(())
                        }
                    } else {
                        Ok(())
                    }
                }
            }
        }
    }

    fn remove_at_pos_if<F>(&mut self, pos: &O, criteria_fn: F) -> Vec<OPair<O, K>>
    where
        F: Fn(&OPair<O, K>) -> bool,
    {
        let mut out = Vec::new();
        let mut empty_after = Vec::new();
        if let Ok(index_found) = self.oc_binary_search(pos) {
            out.append(&mut self[index_found].remove_at_pos_if(pos, &criteria_fn));
            if self[index_found].oc_len() == 0 {
                empty_after.push(index_found);
            }
            if index_found != 0 {
                let mut left = index_found - 1;
                while let Some(Ordering::Equal) = self[left].oc_cmp(pos) {
                    out.append(&mut self[left].remove_at_pos_if(pos, &criteria_fn));
                    if self[left].oc_len() == 0 {
                        empty_after.push(left);
                    }
                    if left == 0 {
                        break;
                    }
                    left -= 1;
                }
            };

            let len = self.len();
            if index_found < len - 1 {
                let mut right = index_found + 1;
                while let Some(Ordering::Equal) = self[right].oc_cmp(pos) {
                    out.append(&mut self[right].remove_at_pos_if(pos, &criteria_fn));
                    if self[right].oc_len() == 0 {
                        empty_after.push(right);
                    }
                    right += 1;
                    if right == len {
                        break;
                    }
                }
            };
        }
        if empty_after.len() != 0 {
            del_multi(self, empty_after.as_ref());
        }
        out
    }

    #[cfg(test)]
    fn recursive_total_len(&self) -> usize {
        let mut total = 0;
        for v in self {
            total += v.recursive_total_len();
        }
        total
    }
}

fn new_subgroup_then_insert<O, K, OC>(
    v: &mut VecDeque<OC>,
    i: usize,
    ins: OPair<O, K>,
) -> Result<(), OCFull<OPair<O, K>>>
where
    O: Ord,
    OC: OrderedContainer<O, K>,
    K: Eq,
{
    if v.len() < MAX_NUM_SUBITEMS {
        v.insert(i, push_to_empty_new_subgroup(ins));
        Ok(())
    } else {
        Err(OCFull(ins))
    }
}

fn push_to_empty_new_subgroup<O, K, OC>(o: OPair<O, K>) -> OC
where
    O: Ord,
    OC: OrderedContainer<O, K>,
    K: Eq,
{
    let mut new_sub = OC::oc_new();
    let _ = new_sub.oc_push_front(o); // guaranteed success
                                      // front or back doesn't matter
    new_sub
}

#[cfg(test)]
mod tests {
    use super::*;
    use crate::test_utils::*;

    #[test]
    fn front_back_cmp() {
        let k = "meh";
        let mut nested_bucket: VecDeque<UnitBucket<usize, &str>> = VecDeque::new();
        assert_eq!(None, nested_bucket.oc_cmp(&9));
        assert!(nested_bucket.oc_front().is_none());
        assert!(nested_bucket.oc_back().is_none());
        assert!(nested_bucket
            .oc_push_front(OPair { pos: 1922, key: k })
            .is_ok());
        assert!(nested_bucket
            .oc_push_front(OPair { pos: 922, key: k })
            .is_ok());
        assert_eq!(Ordering::Greater, nested_bucket.oc_cmp(&9).unwrap());
        assert_eq!(Ordering::Less, nested_bucket.oc_cmp(&2000).unwrap());
        assert_eq!(Ordering::Equal, nested_bucket.oc_cmp(&1000).unwrap());
        assert_eq!(Ordering::Equal, nested_bucket.oc_cmp(&1922).unwrap());
        assert_eq!(Ordering::Equal, nested_bucket.oc_cmp(&922).unwrap());
        assert!(nested_bucket
            .oc_push_front(OPair { pos: 555, key: k })
            .is_ok());
        assert!(nested_bucket
            .oc_push_front(OPair { pos: 555, key: k })
            .is_ok());
        assert!(nested_bucket
            .oc_push_back(OPair { pos: 1922, key: k })
            .is_ok());
        assert_eq!(555, nested_bucket.oc_front().unwrap().pos);
        assert_eq!(1922, nested_bucket.oc_back().unwrap().pos);
        assert_eq!(555, nested_bucket.oc_pop_front().unwrap().pos);
        assert_eq!(1922, nested_bucket.oc_pop_back().unwrap().pos);
    }

    #[test]
    fn many_pops_and_pushes() {
        let k = "meh";
        let mut nested_bucket: VecDeque<UnitBucket<usize, &str>> = VecDeque::new();
        assert!(nested_bucket.front().is_none());
        assert!(nested_bucket.back().is_none());
        assert!(nested_bucket.pop_front().is_none());
        assert!(nested_bucket.pop_back().is_none());

        let mut count = 0;
        for i in 0..MAX_NUM_SUBITEMS * MAX_NUM_SUBITEMS {
            assert!(nested_bucket.oc_push_back(OPair { pos: i, key: k }).is_ok());
            count += 1;
        }
        assert_eq!(count, MAX_NUM_SUBITEMS * MAX_NUM_SUBITEMS);
        assert_eq!(MAX_NUM_SUBITEMS, nested_bucket.len());
        assert_eq!(0, nested_bucket.oc_front().unwrap().pos);
        assert_eq!(count, nested_bucket.oc_back().unwrap().pos + 1);

        let num = 9_000_000;
        assert_eq!(
            Err(OCFull(OPair { pos: num, key: k })),
            nested_bucket.oc_push_back(OPair { pos: num, key: k })
        );

        for (i, unit_bucket) in nested_bucket.iter().enumerate() {
            assert_eq!(unit_bucket.0.len(), MAX_NUM_SUBITEMS);
            assert!(is_sorted(&unit_bucket.0));
            if i < nested_bucket.len() - 1 {
                assert!(
                    nested_bucket[i].oc_back().unwrap() <= nested_bucket[i + 1].oc_front().unwrap()
                );
            }
        }

        let mut nested_bucket: VecDeque<UnitBucket<usize, &str>> = VecDeque::new();
        // reverse
        let mut count = 0;
        for i in (0..MAX_NUM_SUBITEMS * MAX_NUM_SUBITEMS).rev() {
            assert!(nested_bucket
                .oc_push_front(OPair { pos: i, key: k })
                .is_ok());
            count += 1;
        }
        assert_eq!(count, MAX_NUM_SUBITEMS * MAX_NUM_SUBITEMS);
        assert_eq!(MAX_NUM_SUBITEMS, nested_bucket.len());
        assert_eq!(0, nested_bucket.oc_front().unwrap().pos);
        assert_eq!(count, nested_bucket.oc_back().unwrap().pos + 1);

        let num = 0;
        assert_eq!(
            Err(OCFull(OPair { pos: num, key: k })),
            nested_bucket.oc_push_front(OPair { pos: num, key: k })
        );

        for unit_bucket in nested_bucket {
            assert_eq!(unit_bucket.0.len(), MAX_NUM_SUBITEMS);
            assert!(is_sorted(&unit_bucket.0));
        }
    }

    #[test]
    fn binary_search() {
        let k = "meh";
        let nested_bucket: VecDeque<UnitBucket<i32, &str>> = vec![
            vec![
                OPair { pos: 4, key: k },
                OPair { pos: 5, key: k },
                OPair { pos: 6, key: k },
                OPair { pos: 6, key: k },
            ]
            .into(),
            vec![
                OPair { pos: 6, key: k },
                OPair { pos: 6, key: k },
                OPair { pos: 6, key: k },
                OPair { pos: 6, key: k },
            ]
            .into(),
            vec![
                OPair { pos: 6, key: k },
                OPair { pos: 7, key: k },
                OPair { pos: 8, key: k },
                OPair { pos: 9, key: k },
            ]
            .into(),
            vec![
                OPair { pos: 20, key: k },
                OPair { pos: 21, key: k },
                OPair { pos: 22, key: k },
                OPair { pos: 23, key: k },
            ]
            .into(),
            vec![
                OPair { pos: 24, key: k },
                OPair { pos: 25, key: k },
                OPair { pos: 26, key: k },
                OPair { pos: 27, key: k },
            ]
            .into(),
            vec![
                OPair { pos: 91, key: k },
                OPair { pos: 91, key: k },
                OPair { pos: 91, key: k },
                OPair { pos: 95, key: k },
            ]
            .into(),
        ]
        .into();

        assert!(nested_is_sorted(&nested_bucket));
        assert_eq!(Some(Ordering::Equal), nested_bucket.oc_cmp(&7));
        assert_eq!(Some(Ordering::Equal), nested_bucket.oc_cmp(&4));
        assert_eq!(Some(Ordering::Equal), nested_bucket.oc_cmp(&95));
        assert_eq!(Some(Ordering::Greater), nested_bucket.oc_cmp(&3));
        assert_eq!(Some(Ordering::Less), nested_bucket.oc_cmp(&96));

        assert_eq!(Some(Ordering::Less), nested_bucket[0].oc_cmp(&7));
        assert_eq!(Some(Ordering::Less), nested_bucket[1].oc_cmp(&7));
        assert_eq!(Some(Ordering::Equal), nested_bucket[2].oc_cmp(&7));
        assert_eq!(Some(Ordering::Greater), nested_bucket[3].oc_cmp(&7));
        assert_eq!(Some(Ordering::Greater), nested_bucket[4].oc_cmp(&7));
        assert_eq!(Some(Ordering::Greater), nested_bucket[5].oc_cmp(&7));
        assert_eq!(Ok(2), nested_bucket.oc_binary_search(&7));
        assert_eq!(Ok(5), nested_bucket.oc_binary_search(&91));
        assert_eq!(Err(5), nested_bucket.oc_binary_search(&90));
        assert_eq!(Err(6), nested_bucket.oc_binary_search(&96));
        assert_eq!(Err(0), nested_bucket.oc_binary_search(&3));
    }

    #[test]
    fn insert_at_split() {
        let k = "meh";
        let mut nested_bucket: VecDeque<UnitBucket<i32, &str>> = vec![
            vec![
                OPair { pos: 4, key: k },
                OPair { pos: 5, key: k },
                OPair { pos: 6, key: k },
                OPair { pos: 6, key: k },
            ]
            .into(),
            vec![
                OPair { pos: 6, key: k },
                OPair { pos: 6, key: k },
                OPair { pos: 6, key: k },
                OPair { pos: 6, key: k },
            ]
            .into(),
            vec![
                OPair { pos: 6, key: k },
                OPair { pos: 7, key: k },
                OPair { pos: 8, key: k },
                OPair { pos: 9, key: k },
            ]
            .into(),
            vec![
                OPair { pos: 20, key: k },
                OPair { pos: 21, key: k },
                OPair { pos: 22, key: k },
                OPair { pos: 23, key: k },
            ]
            .into(),
            vec![
                OPair { pos: 27, key: k },
                OPair { pos: 28, key: k },
                OPair { pos: 29, key: k },
                OPair { pos: 30, key: k },
            ]
            .into(),
            vec![
                OPair { pos: 91, key: k },
                OPair { pos: 91, key: k },
                OPair { pos: 91, key: k },
                OPair { pos: 95, key: k },
            ]
            .into(),
        ]
        .into();
        let right = nested_bucket.insert_at_split(OPair { pos: 25, key: k });
        assert_eq!(
            right,
            VecDeque::from(vec![
                vec![
                    OPair { pos: 25, key: k },
                    OPair { pos: 27, key: k },
                    OPair { pos: 28, key: k },
                    OPair { pos: 29, key: k },
                    OPair { pos: 30, key: k },
                ]
                .into(),
                vec![
                    OPair { pos: 91, key: k },
                    OPair { pos: 91, key: k },
                    OPair { pos: 91, key: k },
                    OPair { pos: 95, key: k },
                ]
                .into(),
            ])
        );
        assert_eq!(
            nested_bucket,
            VecDeque::from(vec![
                vec![
                    OPair { pos: 4, key: k },
                    OPair { pos: 5, key: k },
                    OPair { pos: 6, key: k },
                    OPair { pos: 6, key: k },
                ]
                .into(),
                vec![
                    OPair { pos: 6, key: k },
                    OPair { pos: 6, key: k },
                    OPair { pos: 6, key: k },
                    OPair { pos: 6, key: k },
                ]
                .into(),
                vec![
                    OPair { pos: 6, key: k },
                    OPair { pos: 7, key: k },
                    OPair { pos: 8, key: k },
                    OPair { pos: 9, key: k },
                ]
                .into(),
                vec![
                    OPair { pos: 20, key: k },
                    OPair { pos: 21, key: k },
                    OPair { pos: 22, key: k },
                    OPair { pos: 23, key: k },
                ]
                .into(),
            ])
        );

        let mut nested_bucket: VecDeque<UnitBucket<i32, &str>> = vec![
            vec![
                OPair { pos: 4, key: k },
                OPair { pos: 5, key: k },
                OPair { pos: 6, key: k },
                OPair { pos: 6, key: k },
            ]
            .into(),
            vec![
                OPair { pos: 6, key: k },
                OPair { pos: 6, key: k },
                OPair { pos: 6, key: k },
                OPair { pos: 6, key: k },
            ]
            .into(),
            vec![
                OPair { pos: 6, key: k },
                OPair { pos: 7, key: k },
                OPair { pos: 8, key: k },
                OPair { pos: 9, key: k },
            ]
            .into(),
            vec![
                OPair { pos: 20, key: k },
                OPair { pos: 21, key: k },
                OPair { pos: 22, key: k },
                OPair { pos: 23, key: k },
            ]
            .into(),
            vec![
                OPair { pos: 27, key: k },
                OPair { pos: 28, key: k },
                OPair { pos: 29, key: k },
                OPair { pos: 30, key: k },
            ]
            .into(),
            vec![
                OPair { pos: 91, key: k },
                OPair { pos: 91, key: k },
                OPair { pos: 91, key: k },
                OPair { pos: 95, key: k },
            ]
            .into(),
        ]
        .into();
        let right = nested_bucket.insert_at_split(OPair { pos: 28, key: k });
        assert_eq!(
            right,
            VecDeque::from(vec![
                vec![
                    OPair { pos: 28, key: k },
                    OPair { pos: 29, key: k },
                    OPair { pos: 30, key: k },
                ]
                .into(),
                vec![
                    OPair { pos: 91, key: k },
                    OPair { pos: 91, key: k },
                    OPair { pos: 91, key: k },
                    OPair { pos: 95, key: k },
                ]
                .into()
            ])
        );
        assert_eq!(
            nested_bucket,
            VecDeque::from(vec![
                vec![
                    OPair { pos: 4, key: k },
                    OPair { pos: 5, key: k },
                    OPair { pos: 6, key: k },
                    OPair { pos: 6, key: k },
                ]
                .into(),
                vec![
                    OPair { pos: 6, key: k },
                    OPair { pos: 6, key: k },
                    OPair { pos: 6, key: k },
                    OPair { pos: 6, key: k },
                ]
                .into(),
                vec![
                    OPair { pos: 6, key: k },
                    OPair { pos: 7, key: k },
                    OPair { pos: 8, key: k },
                    OPair { pos: 9, key: k },
                ]
                .into(),
                vec![
                    OPair { pos: 20, key: k },
                    OPair { pos: 21, key: k },
                    OPair { pos: 22, key: k },
                    OPair { pos: 23, key: k },
                ]
                .into(),
                vec![OPair { pos: 27, key: k }, OPair { pos: 28, key: k },].into(),
            ])
        );

        let mut nested_bucket: VecDeque<UnitBucket<i32, &str>> = vec![
            vec![
                OPair { pos: 4, key: k },
                OPair { pos: 5, key: k },
                OPair { pos: 6, key: k },
                OPair { pos: 6, key: k },
            ]
            .into(),
            vec![
                OPair { pos: 6, key: k },
                OPair { pos: 6, key: k },
                OPair { pos: 6, key: k },
                OPair { pos: 6, key: k },
            ]
            .into(),
            vec![
                OPair { pos: 6, key: k },
                OPair { pos: 7, key: k },
                OPair { pos: 8, key: k },
                OPair { pos: 9, key: k },
            ]
            .into(),
            vec![
                OPair { pos: 20, key: k },
                OPair { pos: 21, key: k },
                OPair { pos: 22, key: k },
                OPair { pos: 23, key: k },
            ]
            .into(),
            vec![
                OPair { pos: 27, key: k },
                OPair { pos: 28, key: k },
                OPair { pos: 29, key: k },
                OPair { pos: 30, key: k },
            ]
            .into(),
            vec![
                OPair { pos: 91, key: k },
                OPair { pos: 91, key: k },
                OPair { pos: 91, key: k },
                OPair { pos: 95, key: k },
            ]
            .into(),
        ]
        .into();
        let right = nested_bucket.insert_at_split(OPair { pos: 3, key: k });
        assert_eq!(
            right,
            VecDeque::from(vec![
                vec![
                    OPair { pos: 3, key: k },
                    OPair { pos: 4, key: k },
                    OPair { pos: 5, key: k },
                    OPair { pos: 6, key: k },
                    OPair { pos: 6, key: k },
                ]
                .into(),
                vec![
                    OPair { pos: 6, key: k },
                    OPair { pos: 6, key: k },
                    OPair { pos: 6, key: k },
                    OPair { pos: 6, key: k },
                ]
                .into(),
                vec![
                    OPair { pos: 6, key: k },
                    OPair { pos: 7, key: k },
                    OPair { pos: 8, key: k },
                    OPair { pos: 9, key: k },
                ]
                .into(),
                vec![
                    OPair { pos: 20, key: k },
                    OPair { pos: 21, key: k },
                    OPair { pos: 22, key: k },
                    OPair { pos: 23, key: k },
                ]
                .into(),
                vec![
                    OPair { pos: 27, key: k },
                    OPair { pos: 28, key: k },
                    OPair { pos: 29, key: k },
                    OPair { pos: 30, key: k },
                ]
                .into(),
                vec![
                    OPair { pos: 91, key: k },
                    OPair { pos: 91, key: k },
                    OPair { pos: 91, key: k },
                    OPair { pos: 95, key: k },
                ]
                .into(),
            ])
        );
        assert_eq!(nested_bucket, VecDeque::from(vec![]));

        let mut nested_bucket: VecDeque<UnitBucket<i32, &str>> = vec![
            vec![
                OPair { pos: 4, key: k },
                OPair { pos: 5, key: k },
                OPair { pos: 6, key: k },
                OPair { pos: 6, key: k },
            ]
            .into(),
            vec![
                OPair { pos: 6, key: k },
                OPair { pos: 6, key: k },
                OPair { pos: 6, key: k },
                OPair { pos: 6, key: k },
            ]
            .into(),
            vec![
                OPair { pos: 6, key: k },
                OPair { pos: 7, key: k },
                OPair { pos: 8, key: k },
                OPair { pos: 9, key: k },
            ]
            .into(),
            vec![
                OPair { pos: 20, key: k },
                OPair { pos: 21, key: k },
                OPair { pos: 22, key: k },
                OPair { pos: 23, key: k },
            ]
            .into(),
            vec![
                OPair { pos: 27, key: k },
                OPair { pos: 28, key: k },
                OPair { pos: 29, key: k },
                OPair { pos: 30, key: k },
            ]
            .into(),
            vec![
                OPair { pos: 91, key: k },
                OPair { pos: 91, key: k },
                OPair { pos: 91, key: k },
                OPair { pos: 95, key: k },
            ]
            .into(),
        ]
        .into();
        let right = nested_bucket.insert_at_split(OPair { pos: 96, key: k });
        assert_eq!(
            right,
            VecDeque::from(vec![vec![OPair { pos: 96, key: k }].into()])
        );
        assert_eq!(
            nested_bucket,
            VecDeque::from(vec![
                vec![
                    OPair { pos: 4, key: k },
                    OPair { pos: 5, key: k },
                    OPair { pos: 6, key: k },
                    OPair { pos: 6, key: k },
                ]
                .into(),
                vec![
                    OPair { pos: 6, key: k },
                    OPair { pos: 6, key: k },
                    OPair { pos: 6, key: k },
                    OPair { pos: 6, key: k },
                ]
                .into(),
                vec![
                    OPair { pos: 6, key: k },
                    OPair { pos: 7, key: k },
                    OPair { pos: 8, key: k },
                    OPair { pos: 9, key: k },
                ]
                .into(),
                vec![
                    OPair { pos: 20, key: k },
                    OPair { pos: 21, key: k },
                    OPair { pos: 22, key: k },
                    OPair { pos: 23, key: k },
                ]
                .into(),
                vec![
                    OPair { pos: 27, key: k },
                    OPair { pos: 28, key: k },
                    OPair { pos: 29, key: k },
                    OPair { pos: 30, key: k },
                ]
                .into(),
                vec![
                    OPair { pos: 91, key: k },
                    OPair { pos: 91, key: k },
                    OPair { pos: 91, key: k },
                    OPair { pos: 95, key: k },
                ]
                .into(),
            ])
        );
    }

    #[test]
    fn insert_sort() {
        let mut nested_bucket: VecDeque<UnitBucket<usize, &str>> = VecDeque::new();
        let randoms = random_vec::<usize>(MAX_NUM_SUBITEMS * MAX_NUM_SUBITEMS);
        for o in randoms {
            if let Err(OCFull(_)) = nested_bucket.insert_sort(OPair { pos: o, key: "meh" }) {
                break;
            }
        }
        assert_eq!(nested_bucket.oc_len(), MAX_NUM_SUBITEMS);
        assert!(nested_is_sorted(&nested_bucket));
        let deep_total = deep_total(&nested_bucket);
        assert!(roughly_within(
            deep_total,
            MAX_NUM_SUBITEMS * MAX_NUM_SUBITEMS / 2,
            10
        )); // stochastic, remove if problematic
    }

    #[test]
    fn remove_eq_cmp_if() {
        let mut nested_tup_bucket: VecDeque<UnitBucket<i32, char>> = vec![
            vec![
                OPair { pos: 126, key: 'j' },
                OPair { pos: 129, key: 'j' },
                OPair { pos: 136, key: 'a' },
                OPair { pos: 136, key: 'f' },
            ]
            .into(),
            vec![
                OPair { pos: 136, key: 'b' },
                OPair { pos: 136, key: 'o' },
                OPair { pos: 136, key: 'm' },
                OPair { pos: 136, key: 'm' },
            ]
            .into(),
            vec![
                OPair { pos: 136, key: 'z' },
                OPair { pos: 136, key: 'x' },
                OPair { pos: 136, key: 'l' },
                OPair { pos: 136, key: 'v' },
            ]
            .into(),
            vec![
                OPair { pos: 136, key: 'j' },
                OPair { pos: 136, key: 'j' },
                OPair { pos: 267, key: 'a' },
                OPair { pos: 277, key: 'f' },
            ]
            .into(),
            vec![
                OPair { pos: 388, key: 'j' },
                OPair { pos: 399, key: 'j' },
                OPair { pos: 499, key: 'a' },
                OPair { pos: 599, key: 'f' },
                OPair { pos: 601, key: 'c' },
                OPair { pos: 682, key: 'n' },
            ]
            .into(),
        ]
        .into();
        assert!(nested_is_sorted(&nested_tup_bucket));
        assert_eq!(5, nested_tup_bucket.oc_len());
        assert_eq!(22, deep_total(&nested_tup_bucket));

        nested_tup_bucket.remove_at_pos_if(&136, |el| el.key > 'e');
        assert_eq!(4, nested_tup_bucket.oc_len());
        assert_eq!(12, deep_total(&nested_tup_bucket));
    }
}
