use std::cmp::Ordering;
use std::collections::HashMap;
use std::hash::Hash;

mod common_refactor;
mod nested_bucket;
#[cfg(test)]
mod test_utils;
mod unit_bucket;

use crate::nested_bucket::NestedBucket;

const MAX_NUM_SUBITEMS: usize = 1024;

pub struct QMap<K: Eq + Hash + Clone, O: Ord + Copy> {
    m: HashMap<K, O>,
    q: Queue<O, K>,
}

impl<K: Eq + Hash + Clone, O: Ord + Copy> QMap<K, O> {
    pub fn new() -> Self {
        Self {
            m: HashMap::new(),
            q: Queue::new(),
        }
    }

    pub fn len(&self) -> usize {
        self.m.len()
    }

    /// Inserts `key` at `order_by` position in the queue. `key` gets
    /// overwritten and given new queue position if exists.
    pub fn upsert(&mut self, key: K, order_by: O) {
        if let Some(old_q_position) = self.m.get(&key) {
            self.q.remove_key_at_position(&key, &old_q_position);
        }
        self.m.insert(key.clone(), order_by);
        self.q.insert(key, order_by);
    }

    /// Deletes `key` and its position in the queue.
    pub fn delete(&mut self, key: &K) {
        if let Some(pos) = self.m.get(&key) {
            self.q.remove_key_at_position(&key, &pos);
        }
        self.m.remove(key);
    }

    /// Tells the position of `key` in the queue. Returns `None` if `key` is
    /// not in store.
    pub fn position(&self, key: &K) -> Option<&O> {
        self.m.get(key)
    }

    pub fn front(&self) -> Option<(&O, &K)> {
        if let Some(pair) = self.q.front() {
            Some((&pair.pos, &pair.key))
        } else {
            None
        }
    }

    pub fn back(&self) -> Option<(&O, &K)> {
        if let Some(pair) = self.q.back() {
            Some((&pair.pos, &pair.key))
        } else {
            None
        }
    }

    pub fn pop_front(&mut self) -> Option<(O, K)> {
        if let Some(pair) = self.q.pop_front() {
            self.m.remove(&pair.key);
            Some((pair.pos, pair.key))
        } else {
            None
        }
    }

    pub fn pop_back(&mut self) -> Option<(O, K)> {
        if let Some(pair) = self.q.pop_back() {
            self.m.remove(&pair.key);
            Some((pair.pos, pair.key))
        } else {
            None
        }
    }

    /// Pops elements from the front of the queue whose position is less than
    /// (inclusive) `until`. Less than means further front.
    pub fn pop_front_until(&mut self, until: &O) -> Vec<(O, K)> {
        let mut out = Vec::new();
        while let Some(front) = self.q.front() {
            if front.pos <= *until {
                let pop = self.q.pop_front().unwrap(); // unwrap safety:
                                                       // this line wouldn't be reached
                                                       // if q.front() is None
                self.m.remove(&pop.key);
                out.push((pop.pos, pop.key));
            } else {
                break;
            }
        }
        out
    }

    /// Pops elements from the back of the queue whose position is greater than
    /// (inclusive) `until`. Greater than means further back.
    pub fn pop_back_until(&mut self, until: &O) -> Vec<(O, K)> {
        let mut out = Vec::new();
        while let Some(back) = self.q.back() {
            if back.pos >= *until {
                let pop = self.q.pop_back().unwrap(); // unwrap safety:
                                                      // this line wouldn't be reached
                                                      // if q.back() is None
                self.m.remove(&pop.key);
                out.push((pop.pos, pop.key));
            } else {
                break;
            }
        }
        out
    }

    pub fn push_front(&mut self, key: K, order_by: O) -> Result<(), InvalidOrder<(O, K)>> {
        if let Some(front) = self.q.hold.front() {
            if order_by > front.pos {
                return Err(InvalidOrder((order_by, key)));
            }
        }
        if let Some(old_q_position) = self.m.get(&key) {
            self.q.remove_key_at_position(&key, &old_q_position);
        }
        self.m.insert(key.clone(), order_by);
        self.q.push_front(key, order_by);
        Ok(())
    }

    pub fn push_back(&mut self, key: K, order_by: O) -> Result<(), InvalidOrder<(O, K)>> {
        if let Some(back) = self.q.hold.back() {
            if order_by < back.pos {
                return Err(InvalidOrder((order_by, key)));
            }
        }
        if let Some(old_q_position) = self.m.get(&key) {
            self.q.remove_key_at_position(&key, &old_q_position);
        }
        self.m.insert(key.clone(), order_by);
        self.q.push_back(key, order_by);
        Ok(())
    }
}

#[derive(Debug, Clone, Eq, PartialEq)]
/// Error type returned by failed pushes. On success, T is consumed and owned
/// by the push destination. If push would result in an out-of-order queue,
/// ownership of T is returned to the caller.
pub struct InvalidOrder<T>(T);
impl<T> std::fmt::Display for InvalidOrder<T> {
    fn fmt(&self, f: &mut std::fmt::Formatter) -> std::fmt::Result {
        write!(f, "push would result in an out-of-order queue")
    }
}

#[derive(Debug, Eq)]
struct OPair<O: Ord, K: Eq> {
    pub pos: O,
    pub key: K,
}
impl<O: Ord, K: Eq> Ord for OPair<O, K> {
    fn cmp(&self, other: &Self) -> Ordering {
        self.pos.cmp(&other.pos)
    }
}
impl<O: Ord, K: Eq> PartialOrd for OPair<O, K> {
    fn partial_cmp(&self, other: &Self) -> Option<Ordering> {
        self.pos.partial_cmp(&other.pos)
    }
}
impl<O: Ord, K: Eq> PartialEq for OPair<O, K> {
    fn eq(&self, other: &Self) -> bool {
        self.pos == other.pos
    }
}

struct Queue<O: Ord, K: Eq> {
    hold: NestedBucket<O, K>,
}

impl<O: Ord, K: Eq> Queue<O, K> {
    pub fn new() -> Self {
        Self {
            hold: NestedBucket::new(),
        }
    }
}

impl<K: Eq, O: Ord + Copy> Queue<O, K> {
    pub fn remove_key_at_position(&mut self, k: &K, pos: &O) {
        let _removed = self.hold.remove(pos, |el| &el.key == k);
        // assert!(_removed.len() <= 1);
    }

    pub fn insert(&mut self, k: K, pos: O) {
        self.hold.insert(OPair { key: k, pos: pos });
    }

    pub fn front(&self) -> Option<&OPair<O, K>> {
        self.hold.front()
    }

    pub fn back(&self) -> Option<&OPair<O, K>> {
        self.hold.back()
    }

    pub fn pop_front(&mut self) -> Option<OPair<O, K>> {
        self.hold.pop_front()
    }

    pub fn pop_back(&mut self) -> Option<OPair<O, K>> {
        self.hold.pop_back()
    }

    // No order check, perform beforehand.
    pub fn push_front(&mut self, k: K, pos: O) {
        self.hold.push_front(OPair { pos: pos, key: k });
    }

    // No order check, perform beforehand.
    pub fn push_back(&mut self, k: K, pos: O) {
        self.hold.push_back(OPair { pos: pos, key: k });
    }
}

trait OrderedContainer<O: Ord, K: Eq> {
    // Trait methods which have the same name as VecDeque method names
    // are prefixed with oc_ to avoid confusion.
    fn oc_new() -> Self;
    fn oc_front(&self) -> Option<&OPair<O, K>>;
    fn oc_back(&self) -> Option<&OPair<O, K>>;
    fn oc_pop_front(&mut self) -> Option<OPair<O, K>>;
    fn oc_pop_back(&mut self) -> Option<OPair<O, K>>;
    fn oc_split_off(&mut self, i: usize) -> Self;
    fn oc_len(&self) -> usize;

    // o < oc_front == Ordering::Less
    // oc_front <= o <= oc_back == Ordering::Equal
    // o > oc_back == Ordering::Greater
    // Returns None if self has no immediate descendant / children.
    // Panics if self contains empty descendants, as they should have
    // been cleaned up upon splits and removals which left a sub-bucket empty.
    fn oc_cmp(&self, o: &O) -> Option<Ordering>;

    // Since this fn relies internally on oc_cmp (which could panic
    // on encountering empty descendant), this fn could panic as well.
    fn oc_binary_search(&self, pos: &O) -> Result<usize, usize>;

    // Pushes and inserts return back the pushed element to the caller upon errors.
    // When Ok, the pushed/inserted element is swallowed.
    // There is no order check inside oc_push so make sure to perform this beforehand.
    fn oc_push_front(&mut self, pair: OPair<O, K>) -> Result<(), OCFull<OPair<O, K>>>;
    fn oc_push_back(&mut self, pair: OPair<O, K>) -> Result<(), OCFull<OPair<O, K>>>;

    // Splits self in such a way that ins can be pushed to the back
    // of the left half (what becomes of `self` after the split)
    // or the front of the right half (returned by the function)
    // and still maintain a sorted order.
    // ins is inserted to the half with less elements.
    fn insert_at_split(&mut self, ins: OPair<O, K>) -> Self;

    // May internally call split_at_inserted.
    fn insert_sort(&mut self, ins: OPair<O, K>) -> Result<(), OCFull<OPair<O, K>>>;

    // Removes all elements at pos that satisfies key_criteria (have it return `true`).
    // key_criteria evaluates each element against a supplied condition.
    // Returns deleted elements.
    fn remove_at_pos_if<F>(&mut self, cmp: &O, key_criteria: F) -> Vec<OPair<O, K>>
    where
        F: Fn(&OPair<O, K>) -> bool;

    #[cfg(test)]
    fn recursive_total_len(&self) -> usize;
}

#[derive(Eq, PartialEq, Debug)]
struct OCFull<T>(T);

#[cfg(test)]
mod tests {
    use super::*;
    use crate::test_utils::*;

    #[test]
    fn api_upsert() {
        let mut map = HashMap::new();
        map.insert("a", 1);
        map.insert("b", 2);
        map.insert("c", 3);
        map.insert("d", 3);
        map.insert("e", 4);
        let mut queue = Queue::new();
        queue.insert("a", 1);
        queue.insert("b", 2);
        queue.insert("c", 3);
        queue.insert("d", 3);
        queue.insert("e", 4);
        let mut qmap = QMap { m: map, q: queue };
        qmap.upsert("c", 9);

        assert_eq!(qmap.m.get("a"), Some(&1));
        assert_eq!(qmap.m.get("b"), Some(&2));
        assert_eq!(qmap.m.get("c"), Some(&9));
        assert_eq!(qmap.m.get("d"), Some(&3));
        assert_eq!(qmap.m.get("e"), Some(&4));

        assert_eq!(qmap.q.pop_front(), Some(OPair { pos: 1, key: "a" }));
        assert_eq!(qmap.q.pop_front(), Some(OPair { pos: 2, key: "b" }));
        assert_eq!(qmap.q.pop_front(), Some(OPair { pos: 3, key: "d" }));
        assert_eq!(qmap.q.pop_front(), Some(OPair { pos: 4, key: "e" }));
        assert_eq!(qmap.q.pop_front(), Some(OPair { pos: 9, key: "c" }));
        assert_eq!(qmap.q.pop_front(), None);
    }

    #[test]
    fn api_pops_and_pushes() {
        let k = "meh";
        let mut qmap = QMap::new();
        for i in 0..3000 {
            assert!(qmap.push_back(k, i).is_ok());
        }
        assert_eq!(1, qmap.len());
        assert_eq!(1, qmap.q.hold.recursive_total_len());

        let mut qmap = QMap::new();
        assert!(qmap.pop_back().is_none());
        assert!(qmap.pop_front().is_none());
        for i in (0..129_000).rev() {
            assert!(qmap.push_front(base64_int(i), i).is_ok());
        }
        for i in 129_000..341_900 {
            assert!(qmap.push_back(base64_int(i), i).is_ok());
        }
        assert_eq!(341_900, qmap.len());
        assert_eq!(341_900, qmap.q.hold.recursive_total_len());

        // oddballs
        qmap.upsert("qu978xfhA".into(), 314_928);
        qmap.upsert("7he9uw3Rf".into(), 223_000);
        qmap.upsert("CMa0Mt3q9".into(), 91_888);
        assert_eq!(341_900 + 3, qmap.len());
        assert_eq!(341_900 + 3, qmap.q.hold.recursive_total_len());

        assert_eq!(Some((0, "AAAAAA".into())), qmap.pop_front());
        assert_eq!(Some((341_900 - 1, "izcFAA".into())), qmap.pop_back());
        let popped_front_len = qmap.pop_front_until(&91_888).len();
        assert_eq!(91_888 + 1 + 1 - 1, popped_front_len);
        let popped_back_len = qmap.pop_back_until(&314_928).len();
        assert_eq!(341_900 - 314_928 - 1 - 1 + 1 + 1, popped_back_len);
        assert_eq!(
            341_900 + 3 - 2 - popped_front_len - popped_back_len,
            qmap.len()
        );
    }
}
