use std::cmp::Ordering;
use std::collections::VecDeque;

use crate::OrderedContainer;

pub(crate) fn cmp_to_non_empty<O, K>(oc: &impl OrderedContainer<O, K>, o: &O) -> Option<Ordering>
where
    O: Ord,
    K: Eq,
{
    let front = if let Some(front) = oc.oc_front() {
        front
    } else {
        panic!("empty descendant")
    };
    let back = if let Some(back) = oc.oc_back() {
        back
    } else {
        panic!("empty descendant")
    };
    if &front.pos > o {
        Some(Ordering::Greater)
    } else if &back.pos < o {
        Some(Ordering::Less)
    } else {
        Some(Ordering::Equal)
    }
}

// O(n) cpu-cache local. Returned Vec is deleted elements.
pub(crate) fn del_multi<T>(v: &mut VecDeque<T>, indexes: &[usize]) -> Vec<T> {
    let len = v.len();
    let mut del = 0;
    for i in 0..len {
        if indexes.contains(&i) {
            del += 1;
        } else if del > 0 {
            v.swap(i - del, i);
        }
    }
    let mut out = Vec::new();
    if del > 0 {
        out = v.drain(len - del..len).collect();
    };
    out
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn multi_del() {
        let mut v: VecDeque<usize> =
            vec![1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 23, 24, 235].into();
        let mut out = del_multi(&mut v, &[1, 0, 1, 2, 8, 9, 10, 11, 100]);
        assert_eq!(v, vec![4, 5, 6, 7, 8, 23, 24, 235]);
        out.sort();
        assert_eq!(out, vec![1, 2, 3, 9, 10, 11, 12]);
    }
}
