use std::io::Cursor;
use std::path::Path;
use std::time::Duration;
use glutin_window::GlutinWindow;
use opengl_graphics::{GlGraphics, OpenGL, Texture, TextureSettings};
use piston::{AdvancedWindow, Button, Key, MouseButton, MouseCursorEvent, Position, PressEvent, ReleaseEvent};
use piston::event_loop::{EventSettings, Events};
use piston::input::{RenderArgs, RenderEvent, UpdateArgs, UpdateEvent};
use piston::window::WindowSettings as PistonWindowSettings;
use winit::dpi::PhysicalSize;

use crate::graphics::Graphics;
use crate::input::Input;
use crate::sprite::{Sprite, SpriteFileFormat};
use crate::window_settings::WindowSettings;


pub struct Window
{
    opengl: OpenGL,
    window_object: GlutinWindow,
    window_settings: WindowSettings,
    input: Input,
    gl: GlGraphics
}

impl Window
{

    pub fn from(window_settings: WindowSettings) -> Window
    {

        let (opengl, window_object) = create_window_from_settings(&window_settings);
        Window {
            opengl,
            window_object,
            gl: GlGraphics::new(opengl),
            input: Input::new(),
            window_settings
        }
    }

    fn set_input_key(&mut self, key: Key, value: bool)
    {
        match key
        {
            Key::Backspace => { self.input.keys[crate::input::Key::Backsspace as usize] = value; }
            Key::Return => { self.input.keys[crate::input::Key::Enter as usize] = value; }
            Key::LShift => { self.input.keys[crate::input::Key::Shift as usize] = value; }
            Key::RShift => { self.input.keys[crate::input::Key::Shift as usize] = value; }
            Key::LCtrl => { self.input.keys[crate::input::Key::Ctrl as usize] = value; }
            Key::RCtrl => { self.input.keys[crate::input::Key::Ctrl as usize] = value; }
            Key::LAlt => { self.input.keys[crate::input::Key::Alt as usize] = value; }
            Key::RAlt => { self.input.keys[crate::input::Key::Alt as usize] = value; }
            Key::Escape => { self.input.keys[crate::input::Key::ESC as usize] = value; }
            Key::Space => { self.input.keys[crate::input::Key::Space as usize] = value; }
            Key::Left => { self.input.keys[crate::input::Key::Left as usize] = value; }
            Key::Up => { self.input.keys[crate::input::Key::Up as usize] = value; }
            Key::Right => { self.input.keys[crate::input::Key::Right as usize] = value; }
            Key::Down => { self.input.keys[crate::input::Key::Down as usize] = value; }
            Key::D0 => { self.input.keys[crate::input::Key::N0 as usize] = value; }
            Key::NumPad0 => { self.input.keys[crate::input::Key::N0 as usize] = value; }
            Key::D1 => { self.input.keys[crate::input::Key::N1 as usize] = value; }
            Key::NumPad1 => { self.input.keys[crate::input::Key::N1 as usize] = value; }
            Key::D2 => { self.input.keys[crate::input::Key::N2 as usize] = value; }
            Key::NumPad2 => { self.input.keys[crate::input::Key::N2 as usize] = value; }
            Key::D3 => { self.input.keys[crate::input::Key::N3 as usize] = value; }
            Key::NumPad3 => { self.input.keys[crate::input::Key::N3 as usize] = value; }
            Key::D4 => { self.input.keys[crate::input::Key::N4 as usize] = value; }
            Key::NumPad4 => { self.input.keys[crate::input::Key::N4 as usize] = value; }
            Key::D5 => { self.input.keys[crate::input::Key::N5 as usize] = value; }
            Key::NumPad5 => { self.input.keys[crate::input::Key::N5 as usize] = value; }
            Key::D6 => { self.input.keys[crate::input::Key::N6 as usize] = value; }
            Key::NumPad6 => { self.input.keys[crate::input::Key::N6 as usize] = value; }
            Key::D7 => { self.input.keys[crate::input::Key::N7 as usize] = value; }
            Key::NumPad7 => { self.input.keys[crate::input::Key::N7 as usize] = value; }
            Key::D8 => { self.input.keys[crate::input::Key::N8 as usize] = value; }
            Key::NumPad8 => { self.input.keys[crate::input::Key::N8 as usize] = value; }
            Key::D9 => { self.input.keys[crate::input::Key::N9 as usize] = value; }
            Key::NumPad9 => { self.input.keys[crate::input::Key::N9 as usize] = value; }
            Key::A => { self.input.keys[crate::input::Key::A as usize] = value; }
            Key::B => { self.input.keys[crate::input::Key::B as usize] = value; }
            Key::C => { self.input.keys[crate::input::Key::C as usize] = value; }
            Key::D => { self.input.keys[crate::input::Key::D as usize] = value; }
            Key::E => { self.input.keys[crate::input::Key::E as usize] = value; }
            Key::F => { self.input.keys[crate::input::Key::F as usize] = value; }
            Key::G => { self.input.keys[crate::input::Key::G as usize] = value; }
            Key::H => { self.input.keys[crate::input::Key::H as usize] = value; }
            Key::I => { self.input.keys[crate::input::Key::I as usize] = value; }
            Key::J => { self.input.keys[crate::input::Key::J as usize] = value; }
            Key::K => { self.input.keys[crate::input::Key::K as usize] = value; }
            Key::L => { self.input.keys[crate::input::Key::L as usize] = value; }
            Key::M => { self.input.keys[crate::input::Key::M as usize] = value; }
            Key::N => { self.input.keys[crate::input::Key::N as usize] = value; }
            Key::O => { self.input.keys[crate::input::Key::O as usize] = value; }
            Key::P => { self.input.keys[crate::input::Key::P as usize] = value; }
            Key::Q => { self.input.keys[crate::input::Key::Q as usize] = value; }
            Key::R => { self.input.keys[crate::input::Key::R as usize] = value; }
            Key::S => { self.input.keys[crate::input::Key::S as usize] = value; }
            Key::T => { self.input.keys[crate::input::Key::T as usize] = value; }
            Key::U => { self.input.keys[crate::input::Key::U as usize] = value; }
            Key::V => { self.input.keys[crate::input::Key::V as usize] = value; }
            Key::W => { self.input.keys[crate::input::Key::W as usize] = value; }
            Key::X => { self.input.keys[crate::input::Key::X as usize] = value; }
            Key::Y => { self.input.keys[crate::input::Key::Y as usize] = value; }
            Key::Z => { self.input.keys[crate::input::Key::Z as usize] = value; }
            Key::F1 => { self.input.keys[crate::input::Key::F1 as usize] = value; }
            Key::F2 => { self.input.keys[crate::input::Key::F2 as usize] = value; }
            Key::F3 => { self.input.keys[crate::input::Key::F3 as usize] = value; }
            Key::F4 => { self.input.keys[crate::input::Key::F4 as usize] = value; }
            Key::F5 => { self.input.keys[crate::input::Key::F5 as usize] = value; }
            Key::F6 => { self.input.keys[crate::input::Key::F6 as usize] = value; }
            Key::F7 => { self.input.keys[crate::input::Key::F7 as usize] = value; }
            Key::F8 => { self.input.keys[crate::input::Key::F8 as usize] = value; }
            Key::F9 => { self.input.keys[crate::input::Key::F9 as usize] = value; }
            Key::F10 => { self.input.keys[crate::input::Key::F10 as usize] = value; }
            Key::F11 => { self.input.keys[crate::input::Key::F11 as usize] = value; }
            Key::F12 => { self.input.keys[crate::input::Key::F12 as usize] = value; }
            _ => {}
        }
    }

    fn set_mouse_button(&mut self, button: MouseButton, value: bool)
    {
        match button {
            MouseButton::Left => { self.input.mouse_left = value; }
            MouseButton::Right => { self.input.mouse_right = value; }
            _ => {}
        }
    }

    pub fn run<F>(mut self, mut on_frame: F) where F: FnMut(&mut WindowSettings, &mut Graphics, &Input) + 'static
    {
        let mut events = Events::new(EventSettings::new());
        let mut dt: f64 = 0.0;

        while let Some(e) = events.next(&mut self.window_object) {

            // input
            if let Some(Button::Keyboard(key)) = e.press_args() { // key down
                self.set_input_key(key, true);
            }
            if let Some(Button::Keyboard(key)) = e.release_args() { // key up
                self.set_input_key(key, false);
            }
            if let Some(Button::Mouse(button)) = e.press_args() { // mouse button down
                self.set_mouse_button(button, true);
            }
            if let Some(Button::Mouse(button)) = e.release_args() { // mouse button up
                self.set_mouse_button(button, false);
            }
            e.mouse_cursor(|pos| { // mouse position
                self.input.mouse_position = [pos[0] as f32, pos[1] as f32];
            });

            // update and render
            if let Some(args) = e.update_args() {
                dt = args.dt;
            }
            if let Some(args) = e.render_args() {

                // get settings before frame

                let last_frame_ws = self.window_settings.clone();

                // do user updating and rendering

                self.gl.draw(args.viewport(), |c, gl| {
                    (on_frame)(&mut self.window_settings, &mut Graphics::new(gl, c, args.window_size[0] as f32, args.window_size[1] as f32, dt), &self.input);
                });

                println!("Previous frame: {:?}, Current frame: {:?}", last_frame_ws.icon.0.len(), self.window_settings.icon.0.len());

                // apply settings if changed

                if last_frame_ws.title != self.window_settings.title {
                    self.window_object.set_title(self.window_settings.title.clone());
                }
                if last_frame_ws.size != self.window_settings.size {
                    self.window_object.set_size(self.window_settings.size);
                    center(&mut self.window_object, self.window_settings.size);
                }
                if last_frame_ws.fullscreen != self.window_settings.fullscreen {
                    self.window_object.ctx.window().set_fullscreen(Option::from(winit::window::Fullscreen::Borderless(None)));
                }
                if last_frame_ws.resizable != self.window_settings.resizable {
                    self.window_object.ctx.window().set_resizable(self.window_settings.resizable);
                }
                if last_frame_ws.icon.0.len() != self.window_settings.icon.0.len() {
                    set_icon(&mut self.window_object, &self.window_settings.icon);
                }

            }

        }
    }

}


fn set_icon(window: &mut GlutinWindow, icon: &(&'static [u8], SpriteFileFormat))
{
    if icon.0.len() > 0
    {
        let icon_sprite = match icon.1 {
            SpriteFileFormat::PNG => { Sprite::new(&icon.0, SpriteFileFormat::PNG) }
            SpriteFileFormat::JPG => { Sprite::new(&icon.0, SpriteFileFormat::JPG) }
        };
        window.ctx.window().set_window_icon(Option::Some(winit::window::Icon::from_rgba(icon_sprite.image.to_vec(), icon_sprite.width, icon_sprite.height).unwrap()));
    }
    else { window.ctx.window().set_window_icon(Option::None); }
}

fn center(window: &mut GlutinWindow, size: [u32; 2])
{
    let screen_size = window.ctx.window().current_monitor().unwrap().size();
    window.set_position( Position { x: ((screen_size.width - size[0]) / 2) as i32, y: ((screen_size.height - size[1]) / 2) as i32 } );
}

fn create_window_from_settings(window_settings: &WindowSettings) -> (OpenGL, GlutinWindow)
{
    // create window with basic settings
    let opengl = OpenGL::V3_2;
    let mut window_object: GlutinWindow = PistonWindowSettings::new(window_settings.title.clone(), window_settings.size.clone())
        .graphics_api(opengl)
        .resizable(window_settings.resizable)
        .fullscreen(window_settings.fullscreen)
        .build().unwrap();

    // set window icon
    set_icon(&mut window_object, &window_settings.icon);

    // center window
    center(&mut window_object, window_settings.size);

    (opengl, window_object)

}
