use std::io::Cursor;
use image::{DynamicImage, RgbaImage};
use opengl_graphics::{Texture, TextureSettings, Format, Filter};


#[derive(Clone)]
pub enum SpriteFileFormat
{
    PNG,
    JPG
}


pub struct Sprite
{
    pub(crate) image: RgbaImage,
    pub(crate) texture: Texture,
    pub width: u32,
    pub height: u32
}

impl Sprite
{
    pub fn new(image_data: &'static [u8], sprite_format: SpriteFileFormat) -> Sprite
    {
        let image_format = match sprite_format {
            SpriteFileFormat::PNG => { image::ImageFormat::Png }
            SpriteFileFormat::JPG => { image::ImageFormat::Jpeg }
        };
        let image = image::load(Cursor::new(image_data), image_format).unwrap().into_rgba8();

        let mut texture_settings = TextureSettings::new();
        texture_settings.set_mag(Filter::Nearest);
        let texture = Texture::from_image(&image, &texture_settings);

        Sprite {
            width: image.width(),
            height: image.height(),
            image,
            texture
        }
    }
}