use std::path::PathBuf;

#[derive(Debug)]
pub struct Entrypoint {
    // original, provided input
    pub entrypoint: String,
    // parsed path to function file
    pub file_path: String,
    // entrypoint function name
    pub function_name: String,
}

impl Entrypoint {
    pub fn new(entrypoint_path: &str) -> Entrypoint {
        let pth_cmpts: Vec<&str> = entrypoint_path.split('.').collect();
        let function = pth_cmpts.last().unwrap();
        let path = &pth_cmpts[..pth_cmpts.len() - 1].join(".");

        // Making assumption this is only ever a *.py file
        let python_path = PathBuf::from(format!("{}.py", path))
            .canonicalize()
            .unwrap();
        if !python_path.as_path().exists() {
            panic!("Entrypoint `{}` not found", entrypoint_path);
        };

        let canonical_path = python_path.to_str().unwrap();

        Entrypoint {
            entrypoint: entrypoint_path.to_owned(),
            file_path: canonical_path.to_owned(),
            function_name: function.to_string(),
        }
    }
}

#[cfg(test)]
mod tests {

    use std::path::PathBuf;

    use super::*;

    #[test]
    fn single() {
        let entrypoint = String::from("../tests/simple/single.s3_list");
        let entry = Entrypoint::new(&entrypoint);
        let canonical_path = PathBuf::from("../tests/simple/single.py")
            .canonicalize()
            .unwrap();
        let path = canonical_path.to_str().unwrap();

        assert_eq!(
            entry.entrypoint,
            String::from("../tests/simple/single.s3_list")
        );
        assert_eq!(entry.file_path, path.to_string());
        assert_eq!(entry.function_name, String::from("s3_list"));
    }
}
