#[cfg(feature = "media")]
use crate::media::extract_generic_filters;
use mcai_worker_sdk::job::JobStatus;
#[cfg(feature = "media")]
use mcai_worker_sdk::prelude::{warn, GenericFilter};
use mcai_worker_sdk::{publish_job_progression, McaiChannel};
use pyo3::prelude::*;
#[cfg(feature = "media")]
use pyo3::types::*;
use std::sync::{Arc, Mutex};

#[pyclass]
pub(crate) struct CallbackHandle {
  pub(crate) channel: Option<McaiChannel>,
  pub(crate) job_id: u64,
  pub(crate) job_status: Arc<Mutex<Option<JobStatus>>>,
}

#[pymethods]
impl CallbackHandle {
  fn publish_job_progression(&self, value: u8) -> bool {
    publish_job_progression(self.channel.clone(), self.job_id, value).is_ok()
  }

  fn is_stopped(&self) -> bool {
    if let Some(channel) = &self.channel {
      channel.lock().unwrap().is_stopped()
    } else {
      false
    }
  }

  fn set_job_status(&mut self, status: &str) -> bool {
    let mut job_status = self.job_status.lock().unwrap();
    *job_status = match status {
      "completed" => Some(JobStatus::Completed),
      "stopped" => Some(JobStatus::Stopped),
      "error" => Some(JobStatus::Error),
      _ => None,
    };

    job_status.is_some()
  }
}

#[pyclass]
#[cfg(feature = "media")]
#[derive(Debug)]
pub(crate) struct StreamDescriptorHandler {}

#[pyclass]
#[cfg(feature = "media")]
#[derive(Debug, Clone)]
pub(crate) struct StreamType {
  video: bool,
  audio: bool,
  data: bool,
}

#[pymethods]
#[cfg(feature = "media")]
impl StreamType {
  #[staticmethod]
  fn new_video() -> Self {
    StreamType {
      video: true,
      audio: false,
      data: false,
    }
  }
  #[staticmethod]
  fn new_audio() -> Self {
    StreamType {
      video: false,
      audio: true,
      data: false,
    }
  }
  #[staticmethod]
  fn new_data() -> Self {
    StreamType {
      video: false,
      audio: false,
      data: true,
    }
  }
  pub(crate) fn is_video(&self) -> bool {
    self.video
  }
  pub(crate) fn is_audio(&self) -> bool {
    self.audio
  }
  pub(crate) fn is_data(&self) -> bool {
    self.data
  }
}

#[pyclass]
#[cfg(feature = "media")]
#[derive(Debug, Clone)]
pub(crate) struct GenericStreamDescriptor {
  pub(crate) index: u32,
  pub(crate) stream_type: StreamType,
  pub(crate) filters: Vec<GenericFilter>,
}

#[pymethods]
#[cfg(feature = "media")]
impl StreamDescriptorHandler {
  #[staticmethod]
  pub fn new_video_stream(index: u32, video_filters: &PyList) -> GenericStreamDescriptor {
    println!("video_filters: {:?}", video_filters);
    let filters = extract_generic_filters(video_filters);

    warn!(
      "StreamDescriptorHandler::new_video_stream ==> filters={:?}",
      filters
    );

    GenericStreamDescriptor {
      index,
      stream_type: StreamType::new_video(),
      filters,
    }
  }

  #[staticmethod]
  pub fn new_audio_stream(index: u32, audio_filters: &PyList) -> GenericStreamDescriptor {
    let filters = extract_generic_filters(audio_filters);

    GenericStreamDescriptor {
      index,
      stream_type: StreamType::new_audio(),
      filters,
    }
  }

  #[staticmethod]
  pub fn new_data_stream(index: u32) -> GenericStreamDescriptor {
    GenericStreamDescriptor {
      index,
      stream_type: StreamType::new_data(),
      filters: vec![],
    }
  }
}
