use crate::media::{extract_generic_filters, PyGenericFilter};
use mcai_worker_sdk::prelude::GenericFilter;
use pyo3::prelude::*;
use pyo3::types::PyDict;

#[test]
pub fn test_generic_filter_from_dict() {
  let python_code = r#"filter = {
                    "name": "crop",
                    "label": "crop_filter",
                    "parameters": {
                       "out_w": "300",
                       "out_h": "200",
                       "x": "50",
                       "y": "30"
                    }
                }"#;

  Python::with_gil(|py| -> PyResult<()> {
    let module = PyModule::from_code(py, python_code, "filter_test.py", "filter_test")?;
    let py_generic_filter: PyGenericFilter = module
      .getattr("filter")
      .unwrap()
      .downcast::<PyDict>()
      .unwrap()
      .into();
    let crop_filter: GenericFilter = py_generic_filter.into();

    assert_eq!("crop".to_string(), crop_filter.name);
    assert_eq!(Some("crop_filter".to_string()), crop_filter.label);
    assert_eq!(
      Some(&"300".to_string()),
      crop_filter.parameters.get("out_w")
    );
    assert_eq!(
      Some(&"200".to_string()),
      crop_filter.parameters.get("out_h")
    );
    assert_eq!(Some(&"50".to_string()), crop_filter.parameters.get("x"));
    assert_eq!(Some(&"30".to_string()), crop_filter.parameters.get("y"));

    Ok(())
  })
  .unwrap();
}

#[test]
pub fn test_generic_filters_from_list() {
  use pyo3::types::PyList;

  let python_code = r#"filters = [{
    "name": "empty_filter",
    "parameters": {}
  },{
      "name": "crop",
      "label": "crop_filter",
      "parameters": {
         "out_w": "300",
         "out_h": "200",
         "x": "50",
         "y": "30"
      }
  }]"#;

  Python::with_gil(|py| -> PyResult<()> {
    let module = PyModule::from_code(py, python_code, "filter_test.py", "filter_test")?;
    let py_list = module
      .getattr("filters")
      .unwrap()
      .downcast::<PyList>()
      .unwrap();
    let generic_filters: Vec<GenericFilter> = extract_generic_filters(py_list);

    let empty_filter = generic_filters.get(0).unwrap();
    assert_eq!("empty_filter".to_string(), empty_filter.name);
    assert!(empty_filter.label.is_none());
    assert!(empty_filter.parameters.is_empty());

    let crop_filter = generic_filters.get(1).unwrap();
    assert_eq!("crop".to_string(), crop_filter.name);
    assert_eq!(Some("crop_filter".to_string()), crop_filter.label);
    assert_eq!(
      Some(&"300".to_string()),
      crop_filter.parameters.get("out_w")
    );
    assert_eq!(
      Some(&"200".to_string()),
      crop_filter.parameters.get("out_h")
    );
    assert_eq!(Some(&"50".to_string()), crop_filter.parameters.get("x"));
    assert_eq!(Some(&"30".to_string()), crop_filter.parameters.get("y"));

    Ok(())
  })
  .unwrap();
}
