use crate::{bat::soc::SoC, error::PwrError, read_number};
use log::debug;
use std::path::{Path, PathBuf};

/// A representation for a battery.
///
/// All battery-related values (such as current charge, voltage, current, etc.) are read via
/// methods. This is to ensure that they are read from sysfs on-demand and always contain
/// up-to-date values.
pub struct Battery {
    path: PathBuf,
}

impl Battery {
    /// Create a new battery instace.
    ///
    /// Expects a path to a battery sysfs entry, such as `/sys/class/power_supply/BAT0`, or
    /// something similar. Note that there isn't a strict naming convention regarding sysfs entries
    /// for batteries, so yours may be named differently.
    pub fn new(path: &Path) -> Result<Battery, PwrError> {
        let path = PathBuf::from(path);
        if !path.exists() {
            return Err(PwrError::NoSysfs(path));
        }

        Ok(Battery { path })
    }

    /// Return a batteries voltage in units of V.
    ///
    /// Can return a [`PwrError`] if e.g. the required sysfs entries don't exist or aren't
    /// accessible.
    pub fn voltage(&self) -> Result<f32, PwrError> {
        let voltage = read_number(self.path.join("voltage_now"))?;

        debug!("Voltage is {:?} uV", voltage);
        Ok(voltage / 1e6)
    }

    /// Returns the batteries state of charge.
    pub fn capacity(&self) -> Result<SoC, PwrError> {
        let charge_cur = read_number(self.path.join("charge_now"))?;
        let charge_full = read_number(self.path.join("charge_full"))?;

        SoC::new(charge_cur / charge_full)
    }

    /// Returns the batteries status.
    pub fn status(&self) -> Result<String, PwrError> {
        // TODO: Implement BatteryStatus enum
        Err(PwrError::NotImplemented("status".into()))
    }

    /// Returns the batteries current in units of A.
    ///
    /// To determine whether the battery is being charged or discharged, refer to [`status`].
    pub fn current(&self) -> Result<f32, PwrError> {
        let current = read_number(self.path.join("current_now"))?;

        debug!("Current is {:?} uA", current);
        Ok(current / 1e6)
    }

    /// Get a default battery.
    pub fn default() -> Result<Self, PwrError> {
        let sysfs = PathBuf::from("/sys/class/power_supply");

        for entry in std::fs::read_dir(&sysfs)? {
            let entry = entry?;
            let entry_name = entry.file_name().to_str().unwrap().to_string();
            debug!("Checking file {}", entry.path().display());
            debug!("File name: {}", entry_name);

            if entry_name.starts_with("BAT")
                || entry_name.starts_with("CMB")
                || entry_name.starts_with("battery")
            {
                return Battery::new(&entry.path());
            }
        }

        Err(PwrError::NoBattery(sysfs))
    }
}

impl std::fmt::Display for Battery {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        write!(
            f,
            "Battery: {}\n\n- Voltage : {:.2} V\n- Current : {:.2} mA\n- Capacity: {}\n",
            self.path.display(),
            self.voltage().unwrap(),
            self.current().unwrap() * 1.0e3,
            self.capacity().unwrap()
        )
    }
}

#[cfg(test)]
mod tests {
    use crate::bat::Battery;
    use crate::tests::dirs;

    #[test]
    fn create() {
        let bat = Battery::new(&dirs::sysfs().join("BAT0"));
        assert!(bat.is_ok());
    }

    #[test]
    fn create_nonexistent_path() {
        let bat = Battery::new(&dirs::sysfs().join("yxz"));
        assert!(bat.is_err());
    }

    #[test]
    fn battery_voltage() {
        let bat = Battery::new(&dirs::sysfs().join("BAT0")).unwrap();
        let voltage = bat.voltage().unwrap();
        assert!(voltage > 0.0);
    }

    #[test]
    fn current() {
        let bat = Battery::new(&dirs::sysfs().join("BAT0")).unwrap();
        let current = bat.current().unwrap();
        assert!(current > 0.0);
    }
}
