//! Runtime glue.

use crate::function::Function;

use serde::{Deserialize, Serialize};
use serde_json::{from_str, to_string};
use std::ffi::{CStr, CString};
use std::os::raw::c_char;

pub fn run<'a, InputType: Deserialize<'a>, OutputType: Serialize>
(node: &dyn Function<'a, InputType, OutputType>, ptr: *mut c_char)
 -> *mut c_char
{
    if ptr.is_null() {
        return 0 as *mut c_char;
    }

    // Read data
    let c_str = unsafe { CStr::from_ptr(ptr) };
    let json_str = match c_str.to_str() { // Check if the content is UTF8-valid
        Ok(s) => s,
        Err(_) => return 1 as *mut c_char
    };

    // Deserialize json
    let json_input: InputType = match from_str(json_str) {
        Ok(v) => v,
        Err(_) => return 2 as *mut c_char
    };

    match node.execute(json_input) {
        Ok(o) => serialize(&*o, '0'),
        Err(e) => serialize(&*e, '1')
    }
}

fn serialize(val: &impl Serialize, is_err: char) -> *mut c_char {
    let output: String = match to_string(val) {
        Ok(s) => format!("{}{}", is_err, s),
        Err(_) => return 3 as *mut c_char
    };

    CString::new(output).unwrap().into_raw()
}