//! Function interface.
//!
//! The function module declares the Function trait and a macro to register types implemeting it.

use crate::error::Error;

use serde::{Deserialize, Serialize};

/// The Function trait is the interface for runnable functions on Punch FaaS.
///
/// # Arguments
///
/// * `'a` - lifetime parameter for serde::Deserialize
/// * `InputType` - input type. It implements the trait serde::Deserialize
/// * `OutputType` - output type. It implements the trait serde::Serialize
pub trait Function<'a, InputType: Deserialize<'a>, OutputType: Serialize> {
    /// Returns a pointer to the computed result.
    ///
    /// # Arguments
    ///
    /// * `input` - function input
    fn execute(&self, input: InputType) -> Result<Box<OutputType>, Box<Error>>;
}

/// Register types implementing the Function trait.
///
/// # Arguments
///
/// * `name` - the type identifier
#[macro_export]
macro_rules! register {
    ( $name:ident ) => {
        #[no_mangle]
        extern fn apply(arg: *mut std::os::raw::c_char) -> *mut std::os::raw::c_char {
            punch_api::run(&$name, arg)
        }
    }
}
