**punch_api is the Punch WebAssembly public API.**  
It consists of interfaces allowing you to write wasm functions able to run on the Punch as FaaS.


# Usage

1. Define the input and output types of your function. They have to respectively implement the `serde::Deserialize` and 
`serde::Serialize` traits.
2. Define your custom function by impleting the `punch_api::Function` trait for the type you define.
3. Register the previous type by calling the `register!` macro on it.
4. You can raise errors by returning a `punch_api::Error` with an error message.

*Et voilà !* By compiling your library to wasm, you are able to run your `Function` on the Punch.

```rust
use punch_api::{Error, Function, register};
use serde::{Deserialize, Serialize};

#[derive(Deserialize)]
pub struct InputValue {
    log: String
}

#[derive(Serialize)]
pub struct OutputValue {
    log_wasm: String,
    size: usize
}

pub struct MyCustomFunction;
impl Function<'_, InputValue, OutputValue> for MyCustomFunction {
    fn execute(&self, input: InputValue) -> Result<Box<OutputValue>, Box<Error>> {
        let input_log_size = input.log.len();
        if input_log_size < 20 {
            let error = Error::new(format!(
                "Invalid log: {}, expected size >= 20, got={}", input.log, input_log_size)
            );
            return Err(Box::new(error))
        }

        let result = OutputValue {
            log_wasm: format!("WASM - received log: *** {} ***", input.log),
            size: input_log_size
        };
        Ok(Box::new(result))
    }
}

register!(MyCustomFunction);
```

## punch_api::Function trait

The `punch_api::Function` is a generic trait taking 3 arguments: a lifetime parameter, an input type and an output type.  
The trait defines a method `execute` that returns a pointer to the result giving the argument `input`.
You may implement your data logic in this method.

```rust
/// The Function trait is the interface for runnable functions on Punch FaaS.
///
/// # Arguments
///
/// * `'a` - lifetime parameter for serde::Deserialize
/// * `InputType` - input type. It implements the trait serde::Deserialize
/// * `OutputType` - output type. It implements the trait serde::Serialize
pub trait Function<'a, InputType: serde::Deserialize<'a>, OutputType: serde::Serialize> {
    /// Returns a pointer to the computed result.
    ///
    /// # Arguments
    ///
    /// * `input` - function input
    fn execute(&self, input: InputType) -> Result<Box<OutputType>, Box<punch_api::Error>>;
}
```