use thiserror::Error;

/// Errors thrown when the server loads its TOML config file

#[derive (Debug, Error)]
pub enum LoadTomlError {
	/// The server's config file is readable by other users, meaning
	/// that the file server module could accidentally serve it to
	/// clients, and expose the server's API key.
	#[error ("Config file has bad permissions mode, it should be octal 0600")]
	ConfigBadPermissions,
	
	/// Wraps `std::io::Error`
	#[error ("I/O")]
	Io (#[from] std::io::Error),
	
	/// Wraps `std::string::FromUtf8Error`
	#[error ("UTF-8")]
	Utf8 (#[from] std::string::FromUtf8Error),
	
	/// Wraps `toml::de::Error`
	#[error ("TOML")]
	Toml (#[from] toml::de::Error),
}

/// Errors thrown when the server is starting up or serving requests

#[derive (Debug, Error)]
pub enum ServerError {
	#[error ("Loading TOML")]
	LoadToml (#[from] LoadTomlError),
	
	#[error ("Loading Handlebars template file")]
	LoadHandlebars (#[from] handlebars::TemplateFileError),
	
	#[error (transparent)]
	FileServer (#[from] super::file_server::errors::FileServerError),
	
	// Hyper stuff
	
	#[error ("Hyper HTTP error")]
	Http (#[from] http::Error),
	
	//#[error ("Hyper invalid header name")]
	//InvalidHeaderName (#[from] hyper::header::InvalidHeaderName),
	
	#[error ("API key invalid")]
	ApiKeyInvalid (http::header::InvalidHeaderValue),
	
	// MessagePack stuff
	
	#[error ("Can't parse wrapped requests in Step 3")]
	CantParseWrappedRequests (rmp_serde::decode::Error),
	
	#[error ("Can't encode PTTH response as MsgPack in Step 5")]
	MessagePackEncodeResponse (rmp_serde::encode::Error),
	
	#[error ("Can't convert Hyper request to PTTH request")]
	CantConvertHyperToPtth (#[from] ptth_core::http_serde::Error),
	
	// Reqwest stuff
	
	#[error ("Can't build HTTP client")]
	CantBuildHttpClient (reqwest::Error),
	
	#[error ("Can't collect non-200 error response body in Step 3")]
	Step3CollectBody (reqwest::Error),
	
	#[error ("Can't collect wrapped requests in Step 3")]
	CantCollectWrappedRequests (reqwest::Error),
	
	#[error ("Error in Step 5, sending response to client through relay")]
	Step5Responding (reqwest::Error),
	
	#[error ("Error in Step 7, getting response from relay after sending response to client")]
	Step7AfterResponse (reqwest::Error),
	
	// UTF-8
	
	#[error ("Step 3 relay response (non-200 OK) was not valid UTF-8")]
	Step3ErrorResponseNotUtf8 (std::string::FromUtf8Error),
	
	#[error (transparent)]
	Other (#[from] anyhow::Error),
}
