#pragma once
#include "ptex-api-export.h"

#include <std_string.h>
#include <stdbool.h>

#ifdef __cplusplus
extern "C" {
#endif

/** Filter types */
enum Ptex__v2_4__PtexFilter__FilterType_e {
    Ptex_PtexFilter_FilterType_f_point = 0,
    Ptex_PtexFilter_FilterType_f_bilinear = 1,
    Ptex_PtexFilter_FilterType_f_box = 2,
    Ptex_PtexFilter_FilterType_f_gaussian = 3,
    Ptex_PtexFilter_FilterType_f_bicubic = 4,
    Ptex_PtexFilter_FilterType_f_bspline = 5,
    Ptex_PtexFilter_FilterType_f_catmullrom = 6,
    Ptex_PtexFilter_FilterType_f_mitchell = 7,
};
typedef unsigned int Ptex_PtexFilter_FilterType;
/** Type of base mesh for which the textures are defined.  A mesh
can be triangle-based (with triangular textures) or quad-based
(with rectangular textures).  */
enum Ptex__v2_4__MeshType_e {
    Ptex_MeshType_mt_triangle = 0,
    Ptex_MeshType_mt_quad = 1,
};
typedef unsigned int Ptex_MeshType;
/** Type of data stored in texture file.  */
enum Ptex__v2_4__DataType_e {
    Ptex_DataType_dt_uint8 = 0,
    Ptex_DataType_dt_uint16 = 1,
    Ptex_DataType_dt_half = 2,
    Ptex_DataType_dt_float = 3,
};
typedef unsigned int Ptex_DataType;
/** How to handle transformation across edges when filtering  */
enum Ptex__v2_4__EdgeFilterMode_e {
    Ptex_EdgeFilterMode_efm_none = 0,
    Ptex_EdgeFilterMode_efm_tanvec = 1,
};
typedef unsigned int Ptex_EdgeFilterMode;
/** How to handle mesh border when filtering.  */
enum Ptex__v2_4__BorderMode_e {
    Ptex_BorderMode_m_clamp = 0,
    Ptex_BorderMode_m_black = 1,
    Ptex_BorderMode_m_periodic = 2,
};
typedef unsigned int Ptex_BorderMode;
/** Edge IDs used in adjacency data in the Ptex::FaceInfo struct.
Edge ID usage for triangle meshes is TBD.  */
enum Ptex__v2_4__EdgeId_e {
    Ptex_EdgeId_e_bottom = 0,
    Ptex_EdgeId_e_right = 1,
    Ptex_EdgeId_e_top = 2,
    Ptex_EdgeId_e_left = 3,
};
typedef unsigned int Ptex_EdgeId;
/** Type of meta data entry.  */
enum Ptex__v2_4__MetaDataType_e {
    Ptex_MetaDataType_mdt_string = 0,
    Ptex_MetaDataType_mdt_int8 = 1,
    Ptex_MetaDataType_mdt_int16 = 2,
    Ptex_MetaDataType_mdt_int32 = 3,
    Ptex_MetaDataType_mdt_float = 4,
    Ptex_MetaDataType_mdt_double = 5,
};
typedef unsigned int Ptex_MetaDataType;

/** Pixel resolution of a given texture.
The resolution is stored in log form: ulog2 = log2(ures), vlog2 = log2(vres)).
Note: negative ulog2 or vlog2 values are reserved for internal use. */
typedef struct Ptex__v2_4__Res_t_s {
    signed char ulog2;
    signed char vlog2;
} PTEX_CPPMM_ALIGN(1) Ptex__v2_4__Res_t;
typedef Ptex__v2_4__Res_t Ptex_Res_t;

/** Information about a face, as stored in the Ptex file header.
The FaceInfo data contains the face resolution and neighboring face
adjacency information as well as a set of flags describing the face.

The adjfaces data member contains the face ids of the four neighboring faces.
The neighbors are accessed in EdgeId order, CCW, starting with the bottom edge.
The adjedges data member contains the corresponding edge id for each neighboring face.

If a face has no neighbor for a given edge, the adjface id should be -1, and the
adjedge id doesn't matter (but is typically zero).

If an adjacent face is a pair of subfaces, the id of the first subface as encountered
in a CCW traversal should be stored as the adjface id. */
typedef struct Ptex__v2_4__FaceInfo_t_s {
    char data[20];
} PTEX_CPPMM_ALIGN(4) Ptex__v2_4__FaceInfo_t;
typedef Ptex__v2_4__FaceInfo_t Ptex_FaceInfo_t;

/** 
   @class PtexMetaData
   @brief Meta data accessor

   Meta data is acquired from PtexTexture and accessed through this interface.
  */
typedef struct Ptex__v2_4__PtexMetaData_t_s {
    char _unused;
} PTEX_CPPMM_ALIGN(8) Ptex__v2_4__PtexMetaData_t;
typedef Ptex__v2_4__PtexMetaData_t Ptex_PtexMetaData_t;

/** 
    @class PtexFaceData
    @brief Per-face texture data accessor

    Per-face texture data is acquired from PtexTexture and accessed
    through this interface.  This interface provides low-level access
    to the data as stored on disk for maximum efficiency.  If this
    isn't needed, face data can be more conveniently read directly
    from PtexTexture.
  */
typedef struct Ptex__v2_4__PtexFaceData_t_s {
    char _unused;
} PTEX_CPPMM_ALIGN(8) Ptex__v2_4__PtexFaceData_t;
typedef Ptex__v2_4__PtexFaceData_t Ptex_PtexFaceData_t;

/** 
   @class PtexTexture
   @brief Interface for reading data from a ptex file

   PtexTexture instances can be acquired via the static open() method, or via
   the PtexCache interface.

   Data access through this interface is returned in v-major order with all data channels interleaved per texel.
  */
typedef struct Ptex__v2_4__PtexTexture_t_s {
    char _unused;
} PTEX_CPPMM_ALIGN(8) Ptex__v2_4__PtexTexture_t;
typedef Ptex__v2_4__PtexTexture_t Ptex_PtexTexture_t;

/** Get most commonly used info in a single call for convenience / efficiency  */
typedef struct Ptex__v2_4__PtexTexture__Info_t_s {
    char data[32];
} PTEX_CPPMM_ALIGN(4) Ptex__v2_4__PtexTexture__Info_t;
typedef Ptex__v2_4__PtexTexture__Info_t Ptex_PtexTexture_Info_t;

/** @class PtexInputHandler
@brief Custom handler interface for intercepting and redirecting Ptex input stream calls

A custom instance of this class can be defined and supplied to the PtexCache class.
Files accessed through the cache will have their input streams redirected through this
interface. */
typedef struct Ptex__v2_4__PtexInputHandler_t_s {
    char _unused;
} PTEX_CPPMM_ALIGN(8) Ptex__v2_4__PtexInputHandler_t;
typedef Ptex__v2_4__PtexInputHandler_t Ptex_PtexInputHandler_t;

/** @class PtexErrorHandler
@brief Custom handler interface redirecting Ptex error messages

A custom instance of this class can be defined and supplied to the PtexCache class.
Files accessed through the cache will have their error streams redirected through this
interface. */
typedef struct Ptex__v2_4__PtexErrorHandler_t_s {
    char _unused;
} PTEX_CPPMM_ALIGN(8) Ptex__v2_4__PtexErrorHandler_t;
typedef Ptex__v2_4__PtexErrorHandler_t Ptex_PtexErrorHandler_t;

/** 
   @class PtexCache
   @brief File-handle and memory cache for reading ptex files

   The PtexCache class allows cached read access to multiple ptex
   files while constraining the open file count and memory usage to
   specified limits.  File and data objects accessed via the cache
   are added back to the cache when their release method is called.
   Released objects are maintained in an LRU list and only destroyed
   when the specified resource limits are exceeded.

   The cache is fully multi-threaded.  Cached data will be shared among
   all threads that have access to the cache, and the data are protected
   with internal locks.  See PtexCache.cpp for details about the caching
   and locking implementation.
  */
typedef struct Ptex__v2_4__PtexCache_t_s {
    char _unused;
} PTEX_CPPMM_ALIGN(8) Ptex__v2_4__PtexCache_t;
typedef Ptex__v2_4__PtexCache_t Ptex_PtexCache_t;

typedef struct Ptex__v2_4__PtexCache__Stats_t_s {
    char data[56];
} PTEX_CPPMM_ALIGN(8) Ptex__v2_4__PtexCache__Stats_t;
typedef Ptex__v2_4__PtexCache__Stats_t Ptex_PtexCache_Stats_t;

/** 
   @class PtexWriter
   @brief Interface for writing data to a ptex file.

   Note: if an alpha channel is specified, then the textures being
   written to the file are expected to have unmultiplied-alpha data.
   Generated mipmaps will be premultiplied by the Ptex library.  On
   read, PtexTexture will (if requested) premultiply all textures by
   alpha when getData is called; by default only reductions are
   premultiplied.  If the source textures are already premultiplied,
   then alphachan can be set to -1 and the library will just leave all
   the data as-is.  The only reason to store unmultiplied-alpha
   textures in the file is to preserve the original texture data for
   later editing. */
typedef struct Ptex__v2_4__PtexWriter_t_s {
    char _unused;
} PTEX_CPPMM_ALIGN(8) Ptex__v2_4__PtexWriter_t;
typedef Ptex__v2_4__PtexWriter_t Ptex_PtexWriter_t;

/** 
   @class PtexFilter
   @brief Interface for filtered sampling of ptex data files.

   PtexFilter instances are obtained by calling one of the particular static methods.  When finished using
   the filter, it must be returned to the library using release().

   To apply the filter to a ptex data file, use the eval() method.
  */
typedef struct Ptex__v2_4__PtexFilter_t_s {
    char _unused;
} PTEX_CPPMM_ALIGN(8) Ptex__v2_4__PtexFilter_t;
typedef Ptex__v2_4__PtexFilter_t Ptex_PtexFilter_t;

/** Choose filter options */
typedef struct Ptex__v2_4__PtexFilter__Options_t_s {
    char data[20];
} PTEX_CPPMM_ALIGN(4) Ptex__v2_4__PtexFilter__Options_t;
typedef Ptex__v2_4__PtexFilter__Options_t Ptex_PtexFilter_Options_t;


/** Default constructor, sets res to 0 (1x1 texel). */
PTEX_CPPMM_API unsigned int Ptex__v2_4__Res_ctor(
    Ptex_Res_t * this_);
#define Ptex_Res_ctor Ptex__v2_4__Res_ctor


/** Constructor. */
PTEX_CPPMM_API unsigned int Ptex__v2_4__Res_from_uv_log2(
    Ptex_Res_t * this_
    , signed char ulog2_
    , signed char vlog2_);
#define Ptex_Res_from_uv_log2 Ptex__v2_4__Res_from_uv_log2


/** Constructor. */
PTEX_CPPMM_API unsigned int Ptex__v2_4__Res_from_value(
    Ptex_Res_t * this_
    , unsigned short value);
#define Ptex_Res_from_value Ptex__v2_4__Res_from_value


/** U resolution in texels. */
PTEX_CPPMM_API unsigned int Ptex__v2_4__Res_u(
    Ptex_Res_t const * this_
    , int * return_);
#define Ptex_Res_u Ptex__v2_4__Res_u


/** V resolution in texels. */
PTEX_CPPMM_API unsigned int Ptex__v2_4__Res_v(
    Ptex_Res_t const * this_
    , int * return_);
#define Ptex_Res_v Ptex__v2_4__Res_v


/** Resolution as a single 16-bit integer value. */
PTEX_CPPMM_API unsigned int Ptex__v2_4__Res_val(
    Ptex_Res_t const * this_
    , unsigned short * return_);
#define Ptex_Res_val Ptex__v2_4__Res_val


/** Total size of specified texture in texels (u * v). */
PTEX_CPPMM_API unsigned int Ptex__v2_4__Res_size(
    Ptex_Res_t const * this_
    , int * return_);
#define Ptex_Res_size Ptex__v2_4__Res_size


/** Comparison operator. */
PTEX_CPPMM_API unsigned int Ptex__v2_4__Res__eq(
    Ptex_Res_t const * this_
    , _Bool * return_
    , Ptex_Res_t const * r);
#define Ptex_Res__eq Ptex__v2_4__Res__eq


/** Comparison operator. */
PTEX_CPPMM_API unsigned int Ptex__v2_4__Res__ne(
    Ptex_Res_t const * this_
    , _Bool * return_
    , Ptex_Res_t const * r);
#define Ptex_Res__ne Ptex__v2_4__Res__ne


/** True if res is >= given res in both u and v directions. */
PTEX_CPPMM_API unsigned int Ptex__v2_4__Res__op_ge(
    Ptex_Res_t const * this_
    , _Bool * return_
    , Ptex_Res_t const * r);
#define Ptex_Res__op_ge Ptex__v2_4__Res__op_ge


/** Get value of resolution with u and v swapped. */
PTEX_CPPMM_API unsigned int Ptex__v2_4__Res_swappeduv(
    Ptex_Res_t const * this_
    , Ptex_Res_t * return_);
#define Ptex_Res_swappeduv Ptex__v2_4__Res_swappeduv


/** Swap the u and v resolution values in place. */
PTEX_CPPMM_API unsigned int Ptex__v2_4__Res_swapuv(
    Ptex_Res_t * this_);
#define Ptex_Res_swapuv Ptex__v2_4__Res_swapuv


/** Clamp the resolution value against the given value. */
PTEX_CPPMM_API unsigned int Ptex__v2_4__Res_clamp(
    Ptex_Res_t * this_
    , Ptex_Res_t const * r);
#define Ptex_Res_clamp Ptex__v2_4__Res_clamp


/** Determine the number of tiles in the u direction for the given tile res. */
PTEX_CPPMM_API unsigned int Ptex__v2_4__Res_ntilesu(
    Ptex_Res_t const * this_
    , int * return_
    , Ptex_Res_t tileres);
#define Ptex_Res_ntilesu Ptex__v2_4__Res_ntilesu


/** Determine the number of tiles in the v direction for the given tile res. */
PTEX_CPPMM_API unsigned int Ptex__v2_4__Res_ntilesv(
    Ptex_Res_t const * this_
    , int * return_
    , Ptex_Res_t tileres);
#define Ptex_Res_ntilesv Ptex__v2_4__Res_ntilesv


/** Determine the total number of tiles for the given tile res. */
PTEX_CPPMM_API unsigned int Ptex__v2_4__Res_ntiles(
    Ptex_Res_t const * this_
    , int * return_
    , Ptex_Res_t tileres);
#define Ptex_Res_ntiles Ptex__v2_4__Res_ntiles


PTEX_CPPMM_API unsigned int Ptex__v2_4__Res_copy(
    Ptex_Res_t * this_
    , Ptex_Res_t const * rhs);
#define Ptex_Res_copy Ptex__v2_4__Res_copy


PTEX_CPPMM_API unsigned int Ptex__v2_4__Res_dtor(
    Ptex_Res_t * this_);
#define Ptex_Res_dtor Ptex__v2_4__Res_dtor


PTEX_CPPMM_API unsigned int Ptex__v2_4__Res_assign(
    Ptex_Res_t * this_
    , Ptex_Res_t * * return_
    , Ptex_Res_t const * rhs);
#define Ptex_Res_assign Ptex__v2_4__Res_assign


/** returns the size of this type in bytes */
PTEX_CPPMM_API unsigned int Ptex__v2_4__FaceInfo_sizeof();
#define Ptex_FaceInfo_sizeof Ptex__v2_4__FaceInfo_sizeof


/** returns the size of this type in bytes */
PTEX_CPPMM_API unsigned int Ptex__v2_4__FaceInfo_alignof();
#define Ptex_FaceInfo_alignof Ptex__v2_4__FaceInfo_alignof


/** Default constructor */
PTEX_CPPMM_API unsigned int Ptex__v2_4__FaceInfo_default(
    Ptex_FaceInfo_t * this_);
#define Ptex_FaceInfo_default Ptex__v2_4__FaceInfo_default


/** Constructor. */
PTEX_CPPMM_API unsigned int Ptex__v2_4__FaceInfo_from_res(
    Ptex_FaceInfo_t * this_
    , Ptex_Res_t res_);
#define Ptex_FaceInfo_from_res Ptex__v2_4__FaceInfo_from_res


/** Constructor. */
PTEX_CPPMM_API unsigned int Ptex__v2_4__FaceInfo_from_res_and_adjacency(
    Ptex_FaceInfo_t * this_
    , Ptex_Res_t res_
    , int adjfaces_[4]
    , int adjedges_[4]
    , _Bool isSubface_);
#define Ptex_FaceInfo_from_res_and_adjacency Ptex__v2_4__FaceInfo_from_res_and_adjacency


/** Access an adjacent edge id.  The eid value must be 0..3. */
PTEX_CPPMM_API unsigned int Ptex__v2_4__FaceInfo_adjedge(
    Ptex_FaceInfo_t const * this_
    , Ptex_EdgeId * return_
    , int eid);
#define Ptex_FaceInfo_adjedge Ptex__v2_4__FaceInfo_adjedge


/** Access an adjacent face id.  The eid value must be 0..3. */
PTEX_CPPMM_API unsigned int Ptex__v2_4__FaceInfo_adjface(
    Ptex_FaceInfo_t const * this_
    , int * return_
    , int eid);
#define Ptex_FaceInfo_adjface Ptex__v2_4__FaceInfo_adjface


/** Determine if face is constant (by checking a flag). */
PTEX_CPPMM_API unsigned int Ptex__v2_4__FaceInfo_isConstant(
    Ptex_FaceInfo_t const * this_
    , _Bool * return_);
#define Ptex_FaceInfo_isConstant Ptex__v2_4__FaceInfo_isConstant


/** Determine if neighborhood of face is constant (by checking a flag). */
PTEX_CPPMM_API unsigned int Ptex__v2_4__FaceInfo_isNeighborhoodConstant(
    Ptex_FaceInfo_t const * this_
    , _Bool * return_);
#define Ptex_FaceInfo_isNeighborhoodConstant Ptex__v2_4__FaceInfo_isNeighborhoodConstant


/** Determine if face has edits in the file (by checking a flag). */
PTEX_CPPMM_API unsigned int Ptex__v2_4__FaceInfo_hasEdits(
    Ptex_FaceInfo_t const * this_
    , _Bool * return_);
#define Ptex_FaceInfo_hasEdits Ptex__v2_4__FaceInfo_hasEdits


/** Determine if face is a subface (by checking a flag). */
PTEX_CPPMM_API unsigned int Ptex__v2_4__FaceInfo_isSubface(
    Ptex_FaceInfo_t const * this_
    , _Bool * return_);
#define Ptex_FaceInfo_isSubface Ptex__v2_4__FaceInfo_isSubface


/** Set the adjfaces data. */
PTEX_CPPMM_API unsigned int Ptex__v2_4__FaceInfo_setadjfaces(
    Ptex_FaceInfo_t * this_
    , int f0
    , int f1
    , int f2
    , int f3);
#define Ptex_FaceInfo_setadjfaces Ptex__v2_4__FaceInfo_setadjfaces


/** Set the adjedges data. */
PTEX_CPPMM_API unsigned int Ptex__v2_4__FaceInfo_setadjedges(
    Ptex_FaceInfo_t * this_
    , int e0
    , int e1
    , int e2
    , int e3);
#define Ptex_FaceInfo_setadjedges Ptex__v2_4__FaceInfo_setadjedges


/** Release resources held by this pointer (pointer becomes invalid). */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexMetaData_release(
    Ptex_PtexMetaData_t * this_);
#define Ptex_PtexMetaData_release Ptex__v2_4__PtexMetaData_release


/** Query number of meta data entries stored in file. */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexMetaData_numKeys(
    Ptex_PtexMetaData_t * this_
    , int * return_);
#define Ptex_PtexMetaData_numKeys Ptex__v2_4__PtexMetaData_numKeys


/** Query the name and type of a meta data entry. */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexMetaData_getKey(
    Ptex_PtexMetaData_t * this_
    , int index
    , char const * * key
    , Ptex_MetaDataType * type);
#define Ptex_PtexMetaData_getKey Ptex__v2_4__PtexMetaData_getKey


/** Query the index and type of a meta data entry by name. */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexMetaData_findKey(
    Ptex_PtexMetaData_t * this_
    , _Bool * return_
    , char const * key
    , int * index
    , Ptex_MetaDataType * type);
#define Ptex_PtexMetaData_findKey Ptex__v2_4__PtexMetaData_findKey


/** Query the value of a given meta data entry.
If the key doesn't exist or the type doesn't match, value is set to null  */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexMetaData_getValueFromKeyChar(
    Ptex_PtexMetaData_t * this_
    , char const * key
    , char const * * value);
#define Ptex_PtexMetaData_getValueFromKeyChar Ptex__v2_4__PtexMetaData_getValueFromKeyChar


/** Query the value of a given meta data entry by index.
If the index is out of range or the type doesn't match, value is set to null  */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexMetaData_getValueFromIndexChar(
    Ptex_PtexMetaData_t * this_
    , int index
    , char const * * value);
#define Ptex_PtexMetaData_getValueFromIndexChar Ptex__v2_4__PtexMetaData_getValueFromIndexChar


/** Query the value of a given meta data entry.
If the key doesn't exist or the type doesn't match, value is set to null  */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexMetaData_getValueFromKeySChar(
    Ptex_PtexMetaData_t * this_
    , char const * key
    , signed char const * * value
    , int * count);
#define Ptex_PtexMetaData_getValueFromKeySChar Ptex__v2_4__PtexMetaData_getValueFromKeySChar


/** Query the value of a given meta data entry by index.
If the index is out of range or the type doesn't match, value is set to null  */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexMetaData_getValueFromIndexSChar(
    Ptex_PtexMetaData_t * this_
    , int index
    , signed char const * * value
    , int * count);
#define Ptex_PtexMetaData_getValueFromIndexSChar Ptex__v2_4__PtexMetaData_getValueFromIndexSChar


/** Query the value of a given meta data entry.
If the key doesn't exist or the type doesn't match, value is set to null  */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexMetaData_getValueFromKeyShort(
    Ptex_PtexMetaData_t * this_
    , char const * key
    , short const * * value
    , int * count);
#define Ptex_PtexMetaData_getValueFromKeyShort Ptex__v2_4__PtexMetaData_getValueFromKeyShort


/** Query the value of a given meta data entry by index.
If the index is out of range or the type doesn't match, value is set to null  */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexMetaData_getValueFromIndexShort(
    Ptex_PtexMetaData_t * this_
    , int index
    , short const * * value
    , int * count);
#define Ptex_PtexMetaData_getValueFromIndexShort Ptex__v2_4__PtexMetaData_getValueFromIndexShort


/** Query the value of a given meta data entry.
If the key doesn't exist or the type doesn't match, value is set to null  */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexMetaData_getValueFromKeyInt(
    Ptex_PtexMetaData_t * this_
    , char const * key
    , int const * * value
    , int * count);
#define Ptex_PtexMetaData_getValueFromKeyInt Ptex__v2_4__PtexMetaData_getValueFromKeyInt


/** Query the value of a given meta data entry by index.
If the index is out of range or the type doesn't match, value is set to null  */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexMetaData_getValueFromIndexInt(
    Ptex_PtexMetaData_t * this_
    , int index
    , int const * * value
    , int * count);
#define Ptex_PtexMetaData_getValueFromIndexInt Ptex__v2_4__PtexMetaData_getValueFromIndexInt


/** Query the value of a given meta data entry.
If the key doesn't exist or the type doesn't match, value is set to null  */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexMetaData_getValueFromKeyFloat(
    Ptex_PtexMetaData_t * this_
    , char const * key
    , float const * * value
    , int * count);
#define Ptex_PtexMetaData_getValueFromKeyFloat Ptex__v2_4__PtexMetaData_getValueFromKeyFloat


/** Query the value of a given meta data entry by index.
If the index is out of range or the type doesn't match, value is set to null  */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexMetaData_getValueFromIndexFloat(
    Ptex_PtexMetaData_t * this_
    , int index
    , float const * * value
    , int * count);
#define Ptex_PtexMetaData_getValueFromIndexFloat Ptex__v2_4__PtexMetaData_getValueFromIndexFloat


/** Query the value of a given meta data entry.
If the key doesn't exist or the type doesn't match, value is set to null  */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexMetaData_getValueFromKeyDouble(
    Ptex_PtexMetaData_t * this_
    , char const * key
    , double const * * value
    , int * count);
#define Ptex_PtexMetaData_getValueFromKeyDouble Ptex__v2_4__PtexMetaData_getValueFromKeyDouble


/** Query the value of a given meta data entry by index.
If the index is out of range or the type doesn't match, value is set to null  */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexMetaData_getValueFromIndexDouble(
    Ptex_PtexMetaData_t * this_
    , int index
    , double const * * value
    , int * count);
#define Ptex_PtexMetaData_getValueFromIndexDouble Ptex__v2_4__PtexMetaData_getValueFromIndexDouble


PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexMetaData__assign(
    Ptex_PtexMetaData_t * this_
    , Ptex_PtexMetaData_t * * return_
    , Ptex_PtexMetaData_t const * rhs);
#define Ptex_PtexMetaData__assign Ptex__v2_4__PtexMetaData__assign


/** Release resources held by this pointer (pointer becomes invalid). */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexFaceData_release(
    Ptex_PtexFaceData_t * this_);
#define Ptex_PtexFaceData_release Ptex__v2_4__PtexFaceData_release


/** True if this data block is constant.  */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexFaceData_isConstant(
    Ptex_PtexFaceData_t * this_
    , _Bool * return_);
#define Ptex_PtexFaceData_isConstant Ptex__v2_4__PtexFaceData_isConstant


/** Resolution of the texture held by this data block.  Note: the
indicated texture res may be larger than 1x1 even if the
texture data is constant.  */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexFaceData_res(
    Ptex_PtexFaceData_t * this_
    , Ptex_Res_t * return_);
#define Ptex_PtexFaceData_res Ptex__v2_4__PtexFaceData_res


/** Read a single texel from the data block.  The texel coordinates, u and v, have
a range of [0..ures-1, 0..vres-1].  Note: this method will work correctly even if
the face is constant or tiled.  */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexFaceData_getPixel(
    Ptex_PtexFaceData_t * this_
    , int u
    , int v
    , void * result);
#define Ptex_PtexFaceData_getPixel Ptex__v2_4__PtexFaceData_getPixel


/** Access the data from this data block.

If the data block is constant, getData will return a pointer to a single texel's data value.

If the data block is tiled, then getData will return null and
the data must be accessed per-tile via the getTile() function.  */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexFaceData_getData(
    Ptex_PtexFaceData_t * this_
    , void * * return_);
#define Ptex_PtexFaceData_getData Ptex__v2_4__PtexFaceData_getData


/** True if this data block is tiled.
If tiled, the data must be access per-tile via getTile().  */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexFaceData_isTiled(
    Ptex_PtexFaceData_t * this_
    , _Bool * return_);
#define Ptex_PtexFaceData_isTiled Ptex__v2_4__PtexFaceData_isTiled


/** Resolution of each tile in this data block.  */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexFaceData_tileRes(
    Ptex_PtexFaceData_t * this_
    , Ptex_Res_t * return_);
#define Ptex_PtexFaceData_tileRes Ptex__v2_4__PtexFaceData_tileRes


/** Access a tile from the data block.  Tiles are accessed in v-major order.  */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexFaceData_getTile(
    Ptex_PtexFaceData_t * this_
    , Ptex_PtexFaceData_t * * return_
    , int tile);
#define Ptex_PtexFaceData_getTile Ptex__v2_4__PtexFaceData_getTile


PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexFaceData__assign(
    Ptex_PtexFaceData_t * this_
    , Ptex_PtexFaceData_t * * return_
    , Ptex_PtexFaceData_t const * rhs);
#define Ptex_PtexFaceData__assign Ptex__v2_4__PtexFaceData__assign


/** Open a ptex file for reading.

If an error occurs, an error message will be stored in the
error string param and a null pointer will be returned.

If the premultiply param is set to true and the texture file has a specified alpha channel,
then all data stored in the file will be multiplied by alpha when read from disk.  If premultiply
is false, then the full-resolution textures will be returned as stored on disk which is assumed
to be unmultiplied.  Reductions (both stored mip-maps and dynamically generated reductions) are
always premultiplied with alpha.  See PtexWriter for more information about alpha channels. */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexTexture_open(
    Ptex_PtexTexture_t * * return_
    , char const * path
    , std_string_t * error
    , _Bool premultiply);
#define Ptex_PtexTexture_open Ptex__v2_4__PtexTexture_open


/** Release resources held by this pointer (pointer becomes invalid). */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexTexture_release(
    Ptex_PtexTexture_t * this_);
#define Ptex_PtexTexture_release Ptex__v2_4__PtexTexture_release


/** Path that file was opened with.  If the file was opened using a search path (via PtexCache),
the the path will be the path as found in the search path.  Otherwise, the path will be
the path as supplied to open.  */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexTexture_path(
    Ptex_PtexTexture_t * this_
    , char const * * return_);
#define Ptex_PtexTexture_path Ptex__v2_4__PtexTexture_path


PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexTexture_getInfo(
    Ptex_PtexTexture_t * this_
    , Ptex_PtexTexture_Info_t * return_);
#define Ptex_PtexTexture_getInfo Ptex__v2_4__PtexTexture_getInfo


/** Type of mesh for which texture data is defined.  */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexTexture_meshType(
    Ptex_PtexTexture_t * this_
    , Ptex_MeshType * return_);
#define Ptex_PtexTexture_meshType Ptex__v2_4__PtexTexture_meshType


/** Type of data stored in file.  */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexTexture_dataType(
    Ptex_PtexTexture_t * this_
    , Ptex_DataType * return_);
#define Ptex_PtexTexture_dataType Ptex__v2_4__PtexTexture_dataType


/** Mode for filtering texture access beyond mesh border.  */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexTexture_uBorderMode(
    Ptex_PtexTexture_t * this_
    , Ptex_BorderMode * return_);
#define Ptex_PtexTexture_uBorderMode Ptex__v2_4__PtexTexture_uBorderMode


/** Mode for filtering texture access beyond mesh border.  */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexTexture_vBorderMode(
    Ptex_PtexTexture_t * this_
    , Ptex_BorderMode * return_);
#define Ptex_PtexTexture_vBorderMode Ptex__v2_4__PtexTexture_vBorderMode


/** Mode for filtering textures across edges.  */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexTexture_edgeFilterMode(
    Ptex_PtexTexture_t * this_
    , Ptex_EdgeFilterMode * return_);
#define Ptex_PtexTexture_edgeFilterMode Ptex__v2_4__PtexTexture_edgeFilterMode


/** Index of alpha channel (if any).  One channel in the file can be flagged to be the alpha channel.
If no channel is acting as the alpha channel, -1 is returned.
See PtexWriter for more details.   */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexTexture_alphaChannel(
    Ptex_PtexTexture_t * this_
    , int * return_);
#define Ptex_PtexTexture_alphaChannel Ptex__v2_4__PtexTexture_alphaChannel


/** Number of channels stored in file.  */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexTexture_numChannels(
    Ptex_PtexTexture_t * this_
    , int * return_);
#define Ptex_PtexTexture_numChannels Ptex__v2_4__PtexTexture_numChannels


/** Number of faces stored in file.  */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexTexture_numFaces(
    Ptex_PtexTexture_t * this_
    , int * return_);
#define Ptex_PtexTexture_numFaces Ptex__v2_4__PtexTexture_numFaces


/** True if the file has edit blocks.  See PtexWriter for more details.  */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexTexture_hasEdits(
    Ptex_PtexTexture_t * this_
    , _Bool * return_);
#define Ptex_PtexTexture_hasEdits Ptex__v2_4__PtexTexture_hasEdits


/** True if the file has mipmaps.  See PtexWriter for more details.  */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexTexture_hasMipMaps(
    Ptex_PtexTexture_t * this_
    , _Bool * return_);
#define Ptex_PtexTexture_hasMipMaps Ptex__v2_4__PtexTexture_hasMipMaps


/** Access meta data.  */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexTexture_getMetaData(
    Ptex_PtexTexture_t * this_
    , Ptex_PtexMetaData_t * * return_);
#define Ptex_PtexTexture_getMetaData Ptex__v2_4__PtexTexture_getMetaData


/** Access resolution and adjacency information about a face.  */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexTexture_getFaceInfo(
    Ptex_PtexTexture_t * this_
    , Ptex_FaceInfo_t const * * return_
    , int faceid);
#define Ptex_PtexTexture_getFaceInfo Ptex__v2_4__PtexTexture_getFaceInfo


/** Access texture data for a face at highest-resolution.

The texture data is copied into the user-supplied buffer.
The buffer must be at least this size (in bytes):
DataSize(dataType()) * numChannels() * getFaceInfo(faceid).res.size().

If a stride is given, then (stride-row_length) bytes will be
skipped after each row.  If stride is zero, then no bytes will
be skipped.  Note: the image can be flipped vertically by using
an appropriate negative stride value.

@param faceid Face index [0..numFaces-1]
@param buffer User-supplied buffer
@param stride Size of each row in user buffer (in bytes) */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexTexture_getDataInto(
    Ptex_PtexTexture_t * this_
    , int faceid
    , void * buffer
    , int stride);
#define Ptex_PtexTexture_getDataInto Ptex__v2_4__PtexTexture_getDataInto


/** Access texture data for a face at a specific resolution.

The specified resolution may be lower than the full resolution
for the face.  If it is lower, then the texture data is
accessed from the stored mip-maps.  If the requested
resolution doesn't match a stored resolution, the desired
resolution will be generated from the nearest available
resolution.

See previous getData() method for interface details. */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexTexture_getDataIntoWithRes(
    Ptex_PtexTexture_t * this_
    , int faceid
    , void * buffer
    , int stride
    , Ptex_Res_t res);
#define Ptex_PtexTexture_getDataIntoWithRes Ptex__v2_4__PtexTexture_getDataIntoWithRes


/** Access texture data for a face at highest-resolution as stored on disk.  */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexTexture_getData(
    Ptex_PtexTexture_t * this_
    , Ptex_PtexFaceData_t * * return_
    , int faceid);
#define Ptex_PtexTexture_getData Ptex__v2_4__PtexTexture_getData


/** Access texture data for a face at a specific resolution as stored on disk.

The specified resolution may be lower (but not higher) than
the full resolution for the face.  If it is lower, then the
texture data is accessed from the stored mip-maps.  If the
requested resolution doesn't match a stored resolution, the
desired resolution will be generated from the nearest
available resolution. */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexTexture_getDataWithRes(
    Ptex_PtexTexture_t * this_
    , Ptex_PtexFaceData_t * * return_
    , int faceid
    , Ptex_Res_t res);
#define Ptex_PtexTexture_getDataWithRes Ptex__v2_4__PtexTexture_getDataWithRes


/** Access a single texel from the highest resolution texture .
The texel data is converted to floating point (integer types
are normalized 0.0 to 1.0).  A subset of the available
channels may be accessed.

@param faceid Face index [0..numFaces-1]
@param u U coordinate [0..ures-1]
@param v V coordinate [0..vres-1]
@param result Result data
@param firstchan First channel to access [0..numChannels-1]
@param nchannels Number of channels to access. */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexTexture_getPixel(
    Ptex_PtexTexture_t * this_
    , int faceid
    , int u
    , int v
    , float * result
    , int firstchan
    , int nchannels);
#define Ptex_PtexTexture_getPixel Ptex__v2_4__PtexTexture_getPixel


/** Access a single texel for a face at a particular resolution.

The specified resolution may be lower (but not higher) than
the full resolution for the face.  If it is lower, then the
texture data is accessed from the stored mip-maps.  If the
requested resolution doesn't match a stored resolution, the
desired resolution will be generated from the nearest
available resolution.

See previous getPixel() method for details. */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexTexture_getPixelWithRes(
    Ptex_PtexTexture_t * this_
    , int faceid
    , int u
    , int v
    , float * result
    , int firstchan
    , int nchannels
    , Ptex_Res_t res);
#define Ptex_PtexTexture_getPixelWithRes Ptex__v2_4__PtexTexture_getPixelWithRes


PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexTexture__assign(
    Ptex_PtexTexture_t * this_
    , Ptex_PtexTexture_t * * return_
    , Ptex_PtexTexture_t const * rhs);
#define Ptex_PtexTexture__assign Ptex__v2_4__PtexTexture__assign


/** returns the size of this type in bytes */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexTexture__Info_sizeof();
#define Ptex_PtexTexture_Info_sizeof Ptex__v2_4__PtexTexture__Info_sizeof


/** returns the size of this type in bytes */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexTexture__Info_alignof();
#define Ptex_PtexTexture_Info_alignof Ptex__v2_4__PtexTexture__Info_alignof


/** Open a file in read mode.
Returns null if there was an error.
If an error occurs, the error string is available via lastError(). */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexInputHandler_open(
    Ptex_PtexInputHandler_t * this_
    , void * * return_
    , char const * path);
#define Ptex_PtexInputHandler_open Ptex__v2_4__PtexInputHandler_open


/** Seek to an absolute byte position in the input stream.  */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexInputHandler_seek(
    Ptex_PtexInputHandler_t * this_
    , void * handle
    , long long pos);
#define Ptex_PtexInputHandler_seek Ptex__v2_4__PtexInputHandler_seek


/** Close a file.  Returns false if an error occurs, and the error
string is available via lastError().   */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexInputHandler_close(
    Ptex_PtexInputHandler_t * this_
    , _Bool * return_
    , void * handle);
#define Ptex_PtexInputHandler_close Ptex__v2_4__PtexInputHandler_close


/** Return the last error message encountered.  */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexInputHandler_lastError(
    Ptex_PtexInputHandler_t * this_
    , char const * * return_);
#define Ptex_PtexInputHandler_lastError Ptex__v2_4__PtexInputHandler_lastError


PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexInputHandler__assign(
    Ptex_PtexInputHandler_t * this_
    , Ptex_PtexInputHandler_t * * return_
    , Ptex_PtexInputHandler_t const * rhs);
#define Ptex_PtexInputHandler__assign Ptex__v2_4__PtexInputHandler__assign


PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexErrorHandler_reportError(
    Ptex_PtexErrorHandler_t * this_
    , char const * error);
#define Ptex_PtexErrorHandler_reportError Ptex__v2_4__PtexErrorHandler_reportError


PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexErrorHandler__assign(
    Ptex_PtexErrorHandler_t * this_
    , Ptex_PtexErrorHandler_t * * return_
    , Ptex_PtexErrorHandler_t const * rhs);
#define Ptex_PtexErrorHandler__assign Ptex__v2_4__PtexErrorHandler__assign


/** Release PtexCache.  Cache will be immediately destroyed and all resources will be released. */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexCache_release(
    Ptex_PtexCache_t * this_);
#define Ptex_PtexCache_release Ptex__v2_4__PtexCache_release


/** Set a search path for finding textures.
Note: if an input handler is installed the search path will be ignored.

@param path colon-delimited search path. */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexCache_setSearchPath(
    Ptex_PtexCache_t * this_
    , char const * path);
#define Ptex_PtexCache_setSearchPath Ptex__v2_4__PtexCache_setSearchPath


/** Query the search path.  Returns string set via setSearchPath.   */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexCache_getSearchPath(
    Ptex_PtexCache_t * this_
    , char const * * return_);
#define Ptex_PtexCache_getSearchPath Ptex__v2_4__PtexCache_getSearchPath


/** Access a texture.  If the specified path was previously accessed
from the cache, then a pointer to the cached texture will be
returned.

If the specified path hasn't been opened yet or was purged
from the cache (via the purge or purgeAll methods) then the
file will be opened.  If the path is relative (i.e. doesn't
begin with a '/') then the search path will be used to locate
the file.

The texture will be accessible until the PtexTexture::release
method is called, at which point the texture will be returned
to the cache.  Once released, the texture may have it's data
pruned (immediately or some time later) to stay within the
maximum cache size.

If the texture could not be opened, null will be returned and
an error string will be set.  If an error were previously
encountered with the file (include the file not being found),
null will be returned and no error string will be set.

@param path File path.  If path is relative, search path will
be used to find the file.

@param error Error string set if texture could not be
opened. */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexCache_get(
    Ptex_PtexCache_t * this_
    , Ptex_PtexTexture_t * * return_
    , char const * path
    , std_string_t * error);
#define Ptex_PtexCache_get Ptex__v2_4__PtexCache_get


/** Remove a texture file from the cache.  If the texture is in use
by another thread, that reference will remain valid and the file
will be purged once it is no longer in use.  This texture
should be released immediately after purging. */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexCache_purge(
    Ptex_PtexCache_t * this_
    , Ptex_PtexTexture_t * texture);
#define Ptex_PtexCache_purge Ptex__v2_4__PtexCache_purge


/** Remove a texture file from the cache by pathname.  The path must
match the full path as opened.  This function will not search
for the file, but if a search path was used, the path must
match the path as found by the search path. */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexCache_purgePath(
    Ptex_PtexCache_t * this_
    , char const * path);
#define Ptex_PtexCache_purgePath Ptex__v2_4__PtexCache_purgePath


/** Remove all texture files from the cache. Textures with
active PtexTexture* handles will remain valid and will be purged
upon release. */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexCache_purgeAll(
    Ptex_PtexCache_t * this_);
#define Ptex_PtexCache_purgeAll Ptex__v2_4__PtexCache_purgeAll


/** Get stats.  */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexCache_getStats(
    Ptex_PtexCache_t * this_
    , Ptex_PtexCache_Stats_t * stats);
#define Ptex_PtexCache_getStats Ptex__v2_4__PtexCache_getStats


PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexCache__assign(
    Ptex_PtexCache_t * this_
    , Ptex_PtexCache_t * * return_
    , Ptex_PtexCache_t const * rhs);
#define Ptex_PtexCache__assign Ptex__v2_4__PtexCache__assign


/** returns the size of this type in bytes */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexCache__Stats_sizeof();
#define Ptex_PtexCache_Stats_sizeof Ptex__v2_4__PtexCache__Stats_sizeof


/** returns the size of this type in bytes */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexCache__Stats_alignof();
#define Ptex_PtexCache_Stats_alignof Ptex__v2_4__PtexCache__Stats_alignof


/** Open a new texture file for writing.
@param path Path to file.
@param mt Type of mesh for which the textures are defined.
@param dt Type of data stored within file.
@param nchannels Number of data channels.
@param alphachan Index of alpha channel, [0..nchannels-1] or -1 if no alpha channel is present.
@param nfaces Number of faces in mesh.
@param error String containing error message if open failed.
@param genmipmaps Specify true if mipmaps should be generated. */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexWriter_open(
    Ptex_PtexWriter_t * * return_
    , char const * path
    , Ptex_MeshType mt
    , Ptex_DataType dt
    , int nchannels
    , int alphachan
    , int nfaces
    , std_string_t * error
    , _Bool genmipmaps);
#define Ptex_PtexWriter_open Ptex__v2_4__PtexWriter_open


/** Open an existing texture file for writing.

If the incremental param is specified as true, then data
values written to the file are appended to the file as "edit
blocks".  This is the fastest way to write data to the file, but
edit blocks are slower to read back, and they have no mipmaps so
filtering can be inefficient.

If incremental is false, then the edits are applied to the
file and the entire file is regenerated on close as if it were
written all at once with open().

If the file doesn't exist it will be created and written as if
open() were used.  If the file exists, the mesh type, data
type, number of channels, alpha channel, and number of faces
must agree with those stored in the file. */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexWriter_edit(
    Ptex_PtexWriter_t * * return_
    , char const * path
    , _Bool incremental
    , Ptex_MeshType mt
    , Ptex_DataType dt
    , int nchannels
    , int alphachan
    , int nfaces
    , std_string_t * error
    , _Bool genmipmaps);
#define Ptex_PtexWriter_edit Ptex__v2_4__PtexWriter_edit


/** Apply edits to a file.

If a file has pending edits, the edits will be applied and the
file will be regenerated with no edits.  This is equivalent to
calling edit() with incremental set to false.  The advantage
is that the file attributes such as mesh type, data type,
etc., don't need to be known in advance. */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexWriter_applyEdits(
    _Bool * return_
    , char const * path
    , std_string_t * error);
#define Ptex_PtexWriter_applyEdits Ptex__v2_4__PtexWriter_applyEdits


/** Release resources held by this pointer (pointer becomes invalid).  */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexWriter_release(
    Ptex_PtexWriter_t * this_);
#define Ptex_PtexWriter_release Ptex__v2_4__PtexWriter_release


/** Set border modes  */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexWriter_setBorderModes(
    Ptex_PtexWriter_t * this_
    , Ptex_BorderMode uBorderMode
    , Ptex_BorderMode vBorderMode);
#define Ptex_PtexWriter_setBorderModes Ptex__v2_4__PtexWriter_setBorderModes


/** Set edge filter mode  */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexWriter_setEdgeFilterMode(
    Ptex_PtexWriter_t * this_
    , Ptex_EdgeFilterMode edgeFilterMode);
#define Ptex_PtexWriter_setEdgeFilterMode Ptex__v2_4__PtexWriter_setEdgeFilterMode


/** Write a string as meta data.  Both the key and string params must be null-terminated strings.  */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexWriter_writeMetaString(
    Ptex_PtexWriter_t * this_
    , char const * key
    , char const * string);
#define Ptex_PtexWriter_writeMetaString Ptex__v2_4__PtexWriter_writeMetaString


/** Write an array of signed 8-bit integers as meta data.  The key must be a null-terminated string.  */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexWriter_writeMetaChar(
    Ptex_PtexWriter_t * this_
    , char const * key
    , signed char const * value
    , int count);
#define Ptex_PtexWriter_writeMetaChar Ptex__v2_4__PtexWriter_writeMetaChar


/** Write an array of signed 16-bit integers as meta data.  The key must be a null-terminated string.  */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexWriter_writeMetaShort(
    Ptex_PtexWriter_t * this_
    , char const * key
    , short const * value
    , int count);
#define Ptex_PtexWriter_writeMetaShort Ptex__v2_4__PtexWriter_writeMetaShort


/** Write an array of signed 32-bit integers as meta data.  The key must be a null-terminated string.  */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexWriter_writeMetaInt(
    Ptex_PtexWriter_t * this_
    , char const * key
    , int const * value
    , int count);
#define Ptex_PtexWriter_writeMetaInt Ptex__v2_4__PtexWriter_writeMetaInt


/** Write an array of signed 32-bit floats as meta data.  The key must be a null-terminated string.  */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexWriter_writeMetaFloat(
    Ptex_PtexWriter_t * this_
    , char const * key
    , float const * value
    , int count);
#define Ptex_PtexWriter_writeMetaFloat Ptex__v2_4__PtexWriter_writeMetaFloat


/** Write an array of signed 32-bit doubles as meta data.  The key must be a null-terminated string.  */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexWriter_writeMetaDouble(
    Ptex_PtexWriter_t * this_
    , char const * key
    , double const * value
    , int count);
#define Ptex_PtexWriter_writeMetaDouble Ptex__v2_4__PtexWriter_writeMetaDouble


/** Copy meta data from an existing meta data block.  */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexWriter_writeMeta(
    Ptex_PtexWriter_t * this_
    , Ptex_PtexMetaData_t * data);
#define Ptex_PtexWriter_writeMeta Ptex__v2_4__PtexWriter_writeMeta


/** Write texture data for a face.
The data is assumed to be channel-interleaved per texel and stored in v-major order.

@param faceid Face index [0..nfaces-1].
@param info Face resolution and adjacency information.
@param data Texel data.
@param stride Distance between rows, in bytes (if zero, data is assumed packed).

If an error is encountered while writing, false is returned and an error message can be
retrieved when close is called. */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexWriter_writeFace(
    Ptex_PtexWriter_t * this_
    , _Bool * return_
    , int faceid
    , Ptex_FaceInfo_t const * info
    , void const * data
    , int stride);
#define Ptex_PtexWriter_writeFace Ptex__v2_4__PtexWriter_writeFace


/** Write constant texture data for a face.
The data is written as a single constant texel value.  Note: the resolution specified in the
info param may indicate a resolution greater than 1x1 and the value will be preserved when
reading.  This is useful to indicate a texture's logical resolution even when the data is
constant.  */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexWriter_writeConstantFace(
    Ptex_PtexWriter_t * this_
    , _Bool * return_
    , int faceid
    , Ptex_FaceInfo_t const * info
    , void const * data);
#define Ptex_PtexWriter_writeConstantFace Ptex__v2_4__PtexWriter_writeConstantFace


/** Close the file.  This operation can take some time if mipmaps are being generated or if there
are many edit blocks.  If an error occurs while writing, false is returned and an error string
is written into the error parameter.  */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexWriter_close(
    Ptex_PtexWriter_t * this_
    , _Bool * return_
    , std_string_t * error);
#define Ptex_PtexWriter_close Ptex__v2_4__PtexWriter_close


PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexWriter__assign(
    Ptex_PtexWriter_t * this_
    , Ptex_PtexWriter_t * * return_
    , Ptex_PtexWriter_t const * rhs);
#define Ptex_PtexWriter__assign Ptex__v2_4__PtexWriter__assign


PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexFilter_getFilter(
    Ptex_PtexFilter_t * * return_
    , Ptex_PtexTexture_t * tx
    , Ptex_PtexFilter_Options_t const * opts);
#define Ptex_PtexFilter_getFilter Ptex__v2_4__PtexFilter_getFilter


/** Release resources held by this pointer (pointer becomes invalid).  */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexFilter_release(
    Ptex_PtexFilter_t * this_);
#define Ptex_PtexFilter_release Ptex__v2_4__PtexFilter_release


/** Apply filter to a ptex data file.

The filter region is a parallelogram centered at the given
(u,v) coordinate with sides defined by two vectors [uw1, vw1]
and [uw2, vw2].  For an axis-aligned rectangle, the vectors
are [uw, 0] and [0, vw].  See \link filterfootprint Filter
Footprint \endlink for details.

@param result Buffer to hold filter result.  Must be large enough to hold nchannels worth of data.
@param firstchan First channel to evaluate [0..tx->numChannels()-1]
@param nchannels Number of channels to evaluate
@param faceid Face index [0..tx->numFaces()-1]
@param u U coordinate, normalized [0..1]
@param v V coordinate, normalized [0..1]
@param uw1 U filter width 1, normalized [0..1]
@param vw1 V filter width 1, normalized [0..1]
@param uw2 U filter width 2, normalized [0..1]
@param vw2 V filter width 2, normalized [0..1]
@param width scale factor for filter width
@param blur amount to add to filter width [0..1] */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexFilter_eval(
    Ptex_PtexFilter_t * this_
    , float * result
    , int firstchan
    , int nchannels
    , int faceid
    , float u
    , float v
    , float uw1
    , float vw1
    , float uw2
    , float vw2
    , float width
    , float blur);
#define Ptex_PtexFilter_eval Ptex__v2_4__PtexFilter_eval


PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexFilter__assign(
    Ptex_PtexFilter_t * this_
    , Ptex_PtexFilter_t * * return_
    , Ptex_PtexFilter_t const * rhs);
#define Ptex_PtexFilter__assign Ptex__v2_4__PtexFilter__assign


/** returns the size of this type in bytes */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexFilter__Options_sizeof();
#define Ptex_PtexFilter_Options_sizeof Ptex__v2_4__PtexFilter__Options_sizeof


/** returns the size of this type in bytes */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexFilter__Options_alignof();
#define Ptex_PtexFilter_Options_alignof Ptex__v2_4__PtexFilter__Options_alignof


/** Constructor - sets defaults */
PTEX_CPPMM_API unsigned int Ptex__v2_4__PtexFilter__Options_ctor(
    Ptex_PtexFilter_Options_t * this_
    , Ptex_PtexFilter_FilterType filter_
    , _Bool lerp_
    , float sharpness_
    , _Bool noedgeblend_);
#define Ptex_PtexFilter_Options_ctor Ptex__v2_4__PtexFilter__Options_ctor


/** Look up name of given mesh type.  */
PTEX_CPPMM_API unsigned int Ptex_v2_4_MeshTypeName(
    char const * * return_
    , Ptex_MeshType mt);
#define Ptex_MeshTypeName Ptex_v2_4_MeshTypeName


/** Look up name of given data type.  */
PTEX_CPPMM_API unsigned int Ptex_v2_4_DataTypeName(
    char const * * return_
    , Ptex_DataType dt);
#define Ptex_DataTypeName Ptex_v2_4_DataTypeName


/** Look up name of given border mode.  */
PTEX_CPPMM_API unsigned int Ptex_v2_4_BorderModeName(
    char const * * return_
    , Ptex_BorderMode m);
#define Ptex_BorderModeName Ptex_v2_4_BorderModeName


/** Look up name of given edge filter mode.  */
PTEX_CPPMM_API unsigned int Ptex_v2_4_EdgeFilterModeName(
    char const * * return_
    , Ptex_EdgeFilterMode m);
#define Ptex_EdgeFilterModeName Ptex_v2_4_EdgeFilterModeName


/** Look up name of given edge ID.  */
PTEX_CPPMM_API unsigned int Ptex_v2_4_EdgeIdName(
    char const * * return_
    , Ptex_EdgeId eid);
#define Ptex_EdgeIdName Ptex_v2_4_EdgeIdName


/** Look up name of given meta data type.  */
PTEX_CPPMM_API unsigned int Ptex_v2_4_MetaDataTypeName(
    char const * * return_
    , Ptex_MetaDataType mdt);
#define Ptex_MetaDataTypeName Ptex_v2_4_MetaDataTypeName


/** Look up size of given data type (in bytes).  */
PTEX_CPPMM_API unsigned int Ptex_v2_4_DataSize(
    int * return_
    , Ptex_DataType dt);
#define Ptex_DataSize Ptex_v2_4_DataSize


/** Look up value of given data type that corresponds to the normalized value of 1.0.  */
PTEX_CPPMM_API unsigned int Ptex_v2_4_OneValue(
    float * return_
    , Ptex_DataType dt);
#define Ptex_OneValue Ptex_v2_4_OneValue


/** Lookup up inverse value of given data type that corresponds to the normalized value of 1.0.  */
PTEX_CPPMM_API unsigned int Ptex_v2_4_OneValueInv(
    float * return_
    , Ptex_DataType dt);
#define Ptex_OneValueInv Ptex_v2_4_OneValueInv


/** Convert a number of data values from the given data type to float.  */
PTEX_CPPMM_API unsigned int Ptex_v2_4_ConvertToFloat(
    float * dst
    , void const * src
    , Ptex_DataType dt
    , int numChannels);
#define Ptex_ConvertToFloat Ptex_v2_4_ConvertToFloat


/** Convert a number of data values from float to the given data type.  */
PTEX_CPPMM_API unsigned int Ptex_v2_4_ConvertFromFloat(
    void * dst
    , float const * src
    , Ptex_DataType dt
    , int numChannels);
#define Ptex_ConvertFromFloat Ptex_v2_4_ConvertFromFloat


#ifdef __cplusplus
}
#endif
